@file:OptIn(DelicateCoroutinesApi::class)

import kotlinx.coroutines.*
import kotlinx.coroutines.flow.onSubscription
import tech.ostack.kform.FormManager
import tech.ostack.kform.LocatedValidationIssue
import tech.ostack.kform.ValidationMode

/**
 * [Validation mode][ValidationMode] representation for use from JavaScript (`"auto" | "manual"`).
 */
public typealias ValidationModeJs = String

internal fun ValidationModeJs.toValidationModeKt(): ValidationMode =
    when (this) {
        "auto" -> ValidationMode.Auto
        "manual" -> ValidationMode.Manual
        else -> error("Invalid validation mode: '$this'.")
    }

/** [Form manager][FormManager] wrapper for use from JavaScript. */
@JsExport
@JsName("FormManager")
public class FormManagerJs(
    formSchema: Any,
    initialValue: Any? = undefined,
    externalContexts: RecordTs<String, Any>? = null,
    validationMode: ValidationModeJs = "auto",
    autoInit: Boolean = true,
) {
    private val formManager =
        if (initialValue == undefined)
            FormManager(
                formSchema.toSchemaKt(),
                jsObjectToMap(externalContexts),
                validationMode.toValidationModeKt(),
                autoInit = autoInit,
            )
        else
            FormManager(
                formSchema.toSchemaKt(),
                initialValue,
                jsObjectToMap(externalContexts),
                validationMode.toValidationModeKt(),
                autoInit = autoInit,
            )

    public fun init(
        externalContexts: RecordTs<String, Any>? = null,
        validationMode: ValidationModeJs = "auto",
    ): CancellablePromise<Nothing?> =
        GlobalScope.cancellablePromise {
            formManager.init(jsObjectToMap(externalContexts), validationMode.toValidationModeKt())
            undefined
        }

    public fun destroy(): CancellablePromise<Nothing?> =
        GlobalScope.cancellablePromise {
            formManager.destroy()
            undefined
        }

    public val autoValidationStatus: AutoValidationStatusJs
        get() = formManager.autoValidationStatus.value.toJs()

    public fun onAutoValidationStatusChange(
        statusChangeHandler: (status: AutoValidationStatusJs) -> Any?,
        onSubscription: (() -> Any?)? = null,
    ): CancellablePromise<() -> CancellablePromise<Nothing?>> =
        GlobalScope.cancellablePromise {
            val subscribed = CompletableDeferred<Unit>()
            val job =
                GlobalScope.launch {
                    formManager.autoValidationStatus
                        .onSubscription {
                            onSubscription?.invoke().maybeAwait()
                            subscribed.complete(Unit)
                        }
                        .collect { statusChangeHandler(it.toJs()).maybeAwait() }
                }
            subscribed.join()
            return@cancellablePromise {
                GlobalScope.cancellablePromise {
                    job.cancelAndJoin()
                    undefined
                }
            }
        }

    public fun setValidationMode(validationMode: ValidationModeJs): CancellablePromise<Nothing?> =
        GlobalScope.cancellablePromise {
            formManager.setValidationMode(validationMode.toValidationModeKt())
            undefined
        }

    public fun isValidPath(path: Any): Boolean = formManager.isValidPath(path.toPathKt())

    public fun schemaInfo(path: Any = AbsolutePathJs.MATCH_ALL): IterableJs<SchemaInfoJs<Any?>> =
        formManager.schemaInfo(path.toPathKt()).toIterableJs {
            @Suppress("UNCHECKED_CAST")
            it.cachedToJs() as SchemaInfoJs<Any?>
        }

    public fun valueInfo(
        path: Any = AbsolutePathJs.MATCH_ALL,
        infoHandler: (infoIterable: AsyncIterableJs<ValueInfoJs<Any?>>) -> Any?,
    ): CancellablePromise<Any?> =
        GlobalScope.cancellablePromise {
            formManager.valueInfo(path.toPathKt()) { infoFlow ->
                infoHandler(infoFlow.toAsyncIterableJs { it.cachedToJs() }).maybeAwait()
            }
        }

    public fun info(
        path: Any = AbsolutePathJs.MATCH_ALL,
        infoHandler: (infoIterable: AsyncIterableJs<InfoJs<Any?>>) -> Any?,
    ): CancellablePromise<Any?> =
        GlobalScope.cancellablePromise {
            formManager.info(path.toPathKt()) { infoFlow ->
                infoHandler(infoFlow.toAsyncIterableJs { it.cachedToJs() }).maybeAwait()
            }
        }

    public fun schema(path: Any = AbsolutePathJs.ROOT): SchemaJs<Any?> =
        @Suppress("UNCHECKED_CAST")
        (formManager.schema(path.toPathKt()).cachedToJs() as SchemaJs<Any?>)

    public fun has(path: Any): CancellablePromise<Boolean> =
        GlobalScope.cancellablePromise { formManager.has(path.toPathKt()) }

    public fun <T> get(
        path: Any = AbsolutePathJs.ROOT,
        valueHandler: (value: T) -> Any?,
    ): CancellablePromise<Any?> =
        GlobalScope.cancellablePromise {
            formManager.get(path.toPathKt()) { value: T -> valueHandler(value).maybeAwait() }
        }

    public fun <T> getClone(path: Any = AbsolutePathJs.ROOT): CancellablePromise<Any?> =
        GlobalScope.cancellablePromise { formManager.getClone<T>(path.toPathKt()) }

    public fun set(path: Any = AbsolutePathJs.ROOT, toSet: Any?): CancellablePromise<Nothing?> =
        GlobalScope.cancellablePromise {
            formManager.set(path.toPathKt(), toSet)
            undefined
        }

    public fun reset(path: Any = AbsolutePathJs.ROOT): CancellablePromise<Nothing?> =
        GlobalScope.cancellablePromise {
            formManager.reset(path.toPathKt())
            undefined
        }

    public fun remove(path: Any): CancellablePromise<Nothing?> =
        GlobalScope.cancellablePromise {
            formManager.remove(path.toPathKt())
            undefined
        }

    public fun getExternalContext(
        externalContextName: String,
        externalContextHandler: (externalContext: Any?) -> Any?,
    ): CancellablePromise<Any?> =
        GlobalScope.cancellablePromise {
            formManager.getExternalContext<Any?, Any?>(externalContextName) {
                externalContextHandler(it).maybeAwait()
            }
        }

    public fun <T> setExternalContext(
        externalContextName: String,
        externalContext: T,
    ): CancellablePromise<T?> =
        GlobalScope.cancellablePromise {
            formManager.setExternalContext(externalContextName, externalContext)
        }

    public fun <T> removeExternalContext(externalContextName: String): CancellablePromise<T?> =
        GlobalScope.cancellablePromise { formManager.removeExternalContext(externalContextName) }

    public fun validate(
        path: Any = AbsolutePathJs.MATCH_ALL,
        issuesHandler: ((issuesIterable: AsyncIterableJs<LocatedValidationIssueJs>) -> Any?)? = null,
    ): CancellablePromise<Any?> =
        GlobalScope.cancellablePromise {
            if (issuesHandler != null)
                formManager.validate(path.toPathKt()) { issuesFlow ->
                    issuesHandler(issuesFlow.toAsyncIterableJs { it.cachedToJs() }).maybeAwait()
                }
            else formManager.validate(path.toPathKt()).map { it.cachedToJs() }.toTypedArray()
        }

    public fun isValid(path: Any = AbsolutePathJs.MATCH_ALL): CancellablePromise<Boolean> =
        GlobalScope.cancellablePromise { formManager.isValid(path.toPathKt()) }

    public fun addExternalIssues(issues: Any): CancellablePromise<Nothing?> =
        GlobalScope.cancellablePromise {
            formManager.addExternalIssues(
                issues.toIterableKt<Any, LocatedValidationIssue> { it.toLocatedValidationIssueKt() }
            )
            undefined
        }

    public fun removeExternalIssues(
        path: Any = AbsolutePathJs.MATCH_ALL,
        code: String? = null,
    ): CancellablePromise<Nothing?> =
        GlobalScope.cancellablePromise {
            formManager.removeExternalIssues(path.toPathKt(), code)
            undefined
        }

    public fun isDirty(path: Any = AbsolutePathJs.ROOT): CancellablePromise<Boolean> =
        GlobalScope.cancellablePromise { formManager.isDirty(path.toPathKt()) }

    public fun isPristine(path: Any = AbsolutePathJs.ROOT): CancellablePromise<Boolean> =
        GlobalScope.cancellablePromise { formManager.isPristine(path.toPathKt()) }

    public fun setDirty(path: Any = AbsolutePathJs.MATCH_ALL): CancellablePromise<Nothing?> =
        GlobalScope.cancellablePromise {
            formManager.setDirty(path.toPathKt())
            undefined
        }

    public fun setPristine(path: Any = AbsolutePathJs.ROOT): CancellablePromise<Nothing?> =
        GlobalScope.cancellablePromise {
            formManager.setPristine(path.toPathKt())
            undefined
        }

    public fun isTouched(path: Any = AbsolutePathJs.ROOT): CancellablePromise<Boolean> =
        GlobalScope.cancellablePromise { formManager.isTouched(path.toPathKt()) }

    public fun isUntouched(path: Any = AbsolutePathJs.ROOT): CancellablePromise<Boolean> =
        GlobalScope.cancellablePromise { formManager.isUntouched(path.toPathKt()) }

    public fun setTouched(path: Any = AbsolutePathJs.MATCH_ALL): CancellablePromise<Nothing?> =
        GlobalScope.cancellablePromise {
            formManager.setTouched(path.toPathKt())
            undefined
        }

    public fun setUntouched(path: Any = AbsolutePathJs.ROOT): CancellablePromise<Nothing?> =
        GlobalScope.cancellablePromise {
            formManager.setUntouched(path.toPathKt())
            undefined
        }

    public fun subscribe(
        path: Any = AbsolutePathJs.MATCH_ALL,
        eventHandler: (event: FormManagerEventJs<Any?>) -> Any?,
        onSubscription: (() -> Any?)? = null,
    ): CancellablePromise<() -> CancellablePromise<Nothing?>> =
        GlobalScope.cancellablePromise {
            val unsubscribe =
                formManager.subscribe(
                    path.toPathKt(),
                    onSubscription?.let { { onSubscription().maybeAwait() } },
                ) {
                    @Suppress("UNCHECKED_CAST")
                    eventHandler(it.cachedToJs() as FormManagerEventJs<Any?>).maybeAwait()
                }
            return@cancellablePromise {
                GlobalScope.cancellablePromise {
                    unsubscribe()
                    undefined
                }
            }
        }
}
