package com.miam.sdk.components.recipeCard

import android.content.Context
import android.util.AttributeSet
import androidx.compose.runtime.Composable
import androidx.compose.runtime.DisposableEffect
import androidx.compose.runtime.LaunchedEffect
import androidx.compose.runtime.collectAsState
import androidx.compose.runtime.getValue
import androidx.compose.runtime.mutableStateOf
import androidx.compose.runtime.remember
import androidx.compose.runtime.setValue
import androidx.compose.ui.platform.AbstractComposeView
import com.miam.core.model.Recipe
import com.miam.core.model.SuggestionsCriteria
import com.miam.core.viewModels.recipe.RecipeContract
import com.miam.core.viewModels.recipe.RecipeViewModel
import com.miam.sdk.components.baseComponent.ManagementResourceState
import com.miam.sdk.components.recipeCard.success.RecipeCardSuccessParams
import com.miam.sdk.di.TemplateDI
import kotlinx.coroutines.delay


class RecipeCard @JvmOverloads constructor(
    context: Context,
    attrs: AttributeSet? = null,
    defStyleAttr: Int = 0
): AbstractComposeView(context, attrs, defStyleAttr) {

    private var vmRecipe: RecipeViewModel = RecipeViewModel()
    internal var isInShelve = true

    fun bind(
        goToDetail: (recipeId: String) -> Unit,
        recipeId: String? = null,
        criteria: SuggestionsCriteria? = null,
        recipe: Recipe? = null
    ) {
        vmRecipe.goToDetail = goToDetail
        if (recipeId != null) {
            vmRecipe.setEvent(RecipeContract.Event.FetchRecipeFromId(recipeId))
        } else if (criteria != null) {
            vmRecipe.setEvent(RecipeContract.Event.FetchRecipeFromCriteria(criteria))
        } else if (recipe != null) {
            vmRecipe.setEvent(RecipeContract.Event.FetchRecipeFromId(recipe.id))
        }
        // Avoid detach listen if vue is dispose by customer
        vmRecipe.registerListeners()
    }


    @Composable
    override fun Content() {
        View(vmRecipe, isInShelve)
    }

    companion object {
        @Composable
        fun View(vmRecipe: RecipeViewModel, isInShelve: Boolean) {

            var timerCompleted by remember(key1 = "recipeCardAnalytics") { mutableStateOf(false) }

            DisposableEffect(Unit) { onDispose {
                vmRecipe.dispose()
                timerCompleted = true
            } }

            LaunchedEffect(Unit) {
                vmRecipe.registerListeners()
            }

            LaunchedEffect(key1 = "recipeCardAnalytics") {
                delay(1000)  // Delay for 1 second
                if (!timerCompleted) {
                    vmRecipe.sendAnalyticsImpression()
                    timerCompleted = true
                }
            }

            val state by vmRecipe.uiState.collectAsState()

            ManagementResourceState(
                resourceState = state.recipeState,
                successView = { recipe ->
                    requireNotNull(recipe)
                    recipe.attributes?.let { recipeAttributes ->
                        TemplateDI.recipeCard.success.view.Content(
                            params = RecipeCardSuccessParams(
                                recipe,
                                recipeAttributes.title,
                                recipeAttributes.mediaUrl,
                                isInShelve,
                                { vmRecipe.setEvent(RecipeContract.Event.GoToDetail) },
                                vmRecipe.guestsInBasket,
                                vmRecipe.currentState.isInCart,
                                vmRecipe.currentState.recipe?.isSponsored == true,
                                if (vmRecipe.currentState.recipe?.isSponsored == true) vmRecipe.currentState.recipe?.relationships?.sponsors?.data?.first()?.attributes?.logoUrl else null,
                                vmRecipe.currentState.likeIsEnable
                            )
                        )
                    }
                },
                loadingView = {
                    TemplateDI.recipeCard.loading.view.Content()
                },
                emptyView = {
                    TemplateDI.recipeCard.empty.view.Content()
                }
            )
        }
    }
}