package com.miam.sdk.components.myMeal

import android.content.Context
import android.util.AttributeSet
import androidx.compose.foundation.layout.Arrangement
import androidx.compose.foundation.layout.Column
import androidx.compose.foundation.layout.fillMaxSize
import androidx.compose.foundation.lazy.LazyColumn
import androidx.compose.foundation.lazy.itemsIndexed
import androidx.compose.runtime.Composable
import androidx.compose.runtime.DisposableEffect
import androidx.compose.runtime.collectAsState
import androidx.compose.runtime.getValue
import androidx.compose.ui.Alignment
import androidx.compose.ui.Modifier
import androidx.compose.ui.platform.AbstractComposeView
import com.miam.core.handler.LogHandler
import com.miam.core.localisation.Localisation
import com.miam.core.model.Recipe
import com.miam.core.viewModels.myMeal.MyMealViewModel
import com.miam.kmm_miam_sdk.android.ressource.Image
import com.miam.sdk.components.baseComponent.ManagementResourceState
import com.miam.sdk.components.baseComponent.emptyPage.EmptyPageParameters
import com.miam.sdk.components.baseComponent.loader.LoaderParameters
import com.miam.sdk.components.myMeal.myMealRecipeCard.MyMealRecipeCard
import com.miam.sdk.components.myMeal.myMealTitle.MyMealTitleParameters
import com.miam.sdk.components.recipeJourney.RecipeJourney
import com.miam.sdk.di.TemplateDI

class MyMeal @JvmOverloads constructor(
    context: Context,
    attrs: AttributeSet? = null,
    defStyleAttr: Int = 0
): AbstractComposeView(context, attrs, defStyleAttr) {
    private val myMealVm = MyMealViewModel()

    private var onButtonClickedWhenEmpty: () -> Unit = {
        LogHandler.error("onButtonClickedWhenEmpty not implemented")
    }

    fun bind(onButtonClickedWhenEmpty: () -> Unit) {
        this.onButtonClickedWhenEmpty = onButtonClickedWhenEmpty
    }

    @Composable
    override fun Content() {
        val state by myMealVm.uiState.collectAsState()

        DisposableEffect(Unit) { onDispose { myMealVm.dispose() } }

        ManagementResourceState(
            resourceState = state.recipes,
            successView = { recipes ->
                requireNotNull(recipes)
                Column(Modifier.fillMaxSize()) {
                    TemplateDI.myMeal.title.view.Content(params = MyMealTitleParameters(recipes.size))
                    LazyColumn(Modifier.fillMaxSize(), verticalArrangement = Arrangement.Top, horizontalAlignment = Alignment.CenterHorizontally) {
                        itemsIndexed(recipes, key = { _, recipe: Recipe -> recipe.id }) { _, recipe ->
                            RecipeJourney.View(recipe = recipe) { recipeId, goToDetail ->
                                MyMealRecipeCard(
                                    recipe = recipe,
                                    goToDetail = { goToDetail(recipeId) }
                                )
                            }
                        }
                    }
                }
            },
            loadingView = {
                TemplateDI.myMeal.loading?.view?.Content(params = LoaderParameters)
                    ?: TemplateDI.defaultViews.loading.view.Content(params = LoaderParameters)
            },
            emptyView = {
                val params = EmptyPageParameters(
                    title = Localisation.MyMeals.noMealIdeaInBasket.localised,
                    haveAnAction = true,
                    image = Image.recipeIdea,
                    actionText = Localisation.MyMeals.goToCatalog.localised
                ) { onButtonClickedWhenEmpty() }

                TemplateDI.myMeal.empty?.view?.Content(params = params)
                    ?: TemplateDI.defaultViews.empty.view.Content(params = params)
            }
        )
    }
}