package com.miam.sdk.components.mealPlanner.meals.toolbar

import androidx.compose.animation.AnimatedVisibility
import androidx.compose.foundation.Image
import androidx.compose.foundation.background
import androidx.compose.foundation.border
import androidx.compose.foundation.clickable
import androidx.compose.foundation.layout.Arrangement
import androidx.compose.foundation.layout.Box
import androidx.compose.foundation.layout.Column
import androidx.compose.foundation.layout.Row
import androidx.compose.foundation.layout.Spacer
import androidx.compose.foundation.layout.fillMaxWidth
import androidx.compose.foundation.layout.height
import androidx.compose.foundation.layout.padding
import androidx.compose.foundation.layout.size
import androidx.compose.foundation.layout.width
import androidx.compose.foundation.layout.wrapContentHeight
import androidx.compose.foundation.shape.RoundedCornerShape
import androidx.compose.material.CircularProgressIndicator
import androidx.compose.material.Divider
import androidx.compose.material.DropdownMenu
import androidx.compose.material.DropdownMenuItem
import androidx.compose.material.Icon
import androidx.compose.material.IconButton
import androidx.compose.material.Text
import androidx.compose.runtime.Composable
import androidx.compose.runtime.getValue
import androidx.compose.runtime.mutableStateOf
import androidx.compose.runtime.remember
import androidx.compose.runtime.setValue
import androidx.compose.ui.Alignment
import androidx.compose.ui.Modifier
import androidx.compose.ui.draw.clip
import androidx.compose.ui.graphics.Color
import androidx.compose.ui.graphics.ColorFilter
import androidx.compose.ui.layout.ContentScale
import androidx.compose.ui.res.painterResource
import androidx.compose.ui.text.TextStyle
import androidx.compose.ui.text.font.FontWeight
import androidx.compose.ui.text.style.TextAlign
import androidx.compose.ui.unit.dp
import androidx.compose.ui.unit.sp
import com.miam.core.base.state.ComponentUiState
import com.miam.core.localisation.Localisation
import com.miam.kmm_miam_sdk.android.R

import com.miam.sdk.components.mealPlanner.form.success.MealPlannerFormSuccessParameters


internal class MealPlannerToolbarImp: MealPlannerToolbar {
    @Composable
    override fun Content(mealPlannerFormParameters: MealPlannerFormSuccessParameters) {

        Box(
            modifier = Modifier
                .fillMaxWidth()
                .background(Color.White),
            contentAlignment = Alignment.BottomStart
        ) {

            Column(
                modifier = Modifier
                    .padding(top = 24.dp)
                    .padding(horizontal = com.miam.kmm_miam_sdk.android.theme.Dimension.lPadding)
            ) {
                Row(
                    modifier = Modifier.fillMaxWidth(),
                    verticalAlignment = Alignment.CenterVertically,
                    horizontalArrangement = Arrangement.SpaceEvenly
                ) {

                    // Budget
                    MealPlannerBudget(
                        modifier = Modifier.width(96.dp),
                        currency = Localisation.Price.currency.localised,
                        budget = mealPlannerFormParameters.budget
                    ) {
                        mealPlannerFormParameters.setBudget(it)
                    }

                    // Number of guests
                    MealPlannerDropDown(
                        leftContent = {
                            Icon(
                                modifier = Modifier
                                    .width(48.dp)
                                    .wrapContentHeight(),
                                painter = painterResource(id = R.drawable.ic_guests),
                                tint = com.miam.kmm_miam_sdk.android.theme.Colors.black,
                                contentDescription = null
                            )
                        },
                        rightContent = {
                            Text(
                                modifier = Modifier
                                    .width(48.dp)
                                    .wrapContentHeight(),
                                text = mealPlannerFormParameters.numberOfGuests.toString(),
                                textAlign = TextAlign.Center,
                                style = TextStyle(
                                    color = com.miam.kmm_miam_sdk.android.theme.Colors.black,
                                    fontSize = 16.sp,
                                    fontWeight = FontWeight.W500
                                )
                            )
                        },
                        onValueChanged = {
                            mealPlannerFormParameters.setNumberOfGuests(it)
                            println("New value set to $it")
                        }
                    )

                    // Number of meals
                    MealPlannerDropDown(
                        leftContent = {
                            Image(
                                modifier = Modifier
                                    .width(48.dp)
                                    .padding(14.dp),
                                contentScale = ContentScale.Fit,
                                painter = painterResource(id = R.drawable.ic_meals),
                                colorFilter = ColorFilter.tint(com.miam.kmm_miam_sdk.android.theme.Colors.black),
                                contentDescription = null
                            )
                        },
                        rightContent = {
                            Text(
                                modifier = Modifier.width(48.dp),
                                text = mealPlannerFormParameters.numberOfMeals.toString(),
                                textAlign = TextAlign.Center,
                                style = TextStyle(
                                    color = com.miam.kmm_miam_sdk.android.theme.Colors.black,
                                    fontSize = 16.sp,
                                    fontWeight = FontWeight.W500
                                )
                            )
                        },
                        onValueChanged = {
                            mealPlannerFormParameters.setNumberOfMeals(it)
                            println("New value set to $it")
                        }
                    )

                    // Validate
                    IconButton(
                        onClick = {
                            mealPlannerFormParameters.submit(
                                mealPlannerFormParameters.budget,
                                mealPlannerFormParameters.numberOfMeals,
                                mealPlannerFormParameters.numberOfGuests
                            )
                        },
                        enabled = mealPlannerFormParameters.budget >= 2 && mealPlannerFormParameters.uiState != ComponentUiState.LOADING,
                        modifier = Modifier
                            .size(48.dp)
                            .clip(RoundedCornerShape(6.dp))
                            .background(com.miam.kmm_miam_sdk.android.theme.Colors.primary)
                    ) {
                        if (mealPlannerFormParameters.uiState == ComponentUiState.LOADING) {
                            CircularProgressIndicator(
                                modifier = Modifier.size(com.miam.kmm_miam_sdk.android.theme.Dimension.mIconHeight),
                                strokeWidth = 2.dp,
                                color = com.miam.kmm_miam_sdk.android.theme.Colors.white
                            )
                        } else {
                            Image(
                                painter = painterResource(com.miam.kmm_miam_sdk.android.ressource.Image.check),
                                contentDescription = "check icon",
                                colorFilter = ColorFilter.tint(com.miam.kmm_miam_sdk.android.theme.Colors.white),
                                modifier = Modifier.size(18.dp)
                            )
                        }
                    }
                }

                AnimatedVisibility(mealPlannerFormParameters.uiState == ComponentUiState.EMPTY) {
                    Column() {
                        Spacer(modifier = Modifier.height(32.dp))
                        Box(
                            modifier = Modifier.background(
                                com.miam.kmm_miam_sdk.android.theme.Colors.miamDangerBackground,
                                RoundedCornerShape(8.dp)
                            )
                        ) {
                            Column(
                                Modifier.padding(16.dp)
                            ) {
                                Icon(
                                    painter = painterResource(id = com.miam.kmm_miam_sdk.android.ressource.Image.alert),
                                    contentDescription = "",
                                    tint = com.miam.kmm_miam_sdk.android.theme.Colors.dangerBudgetText
                                )
                                Spacer(modifier = Modifier.height(8.dp))
                                Text(
                                    text = Localisation.Budget.noResultsForBudget.localised,
                                    color = com.miam.kmm_miam_sdk.android.theme.Colors.dangerBudgetText
                                )
                            }
                        }
                    }

                }
            }
        }
    }

    @Composable
    fun MealPlannerDropDown(
        modifier: Modifier = Modifier,
        leftContent: @Composable () -> Unit,
        rightContent: @Composable () -> Unit,
        onValueChanged: (Int) -> Unit,
    ) {

        val borderColor = Color(0XFFD9DDE1)
        var expanded by remember { mutableStateOf(false) }

        Row(
            modifier = modifier
                .height(48.dp)
                .border(
                    width = 1.dp,
                    color = borderColor,
                    shape = RoundedCornerShape(6.dp)
                )
                .clickable {
                    expanded = !expanded
                },
            verticalAlignment = Alignment.CenterVertically,
        ) {
            leftContent()
            Divider(modifier = Modifier.size(width = 1.dp, height = 48.dp), color = borderColor)
            rightContent()

            var selectedItem by remember { mutableStateOf("1") }
            val numbers = (1..9).map { it.toString() }

            DropdownMenu(
                modifier = Modifier.width(96.dp),
                expanded = expanded,
                onDismissRequest = { expanded = false },
            ) {
                numbers.forEach { number ->
                    DropdownMenuItem(onClick = {
                        selectedItem = number
                        onValueChanged(selectedItem.toInt())
                        expanded = false
                    }) {
                        Text(text = number)
                    }
                }
            }
        }
    }
}