package com.miam.sdk.components.favoritePage

import android.content.Context
import android.util.AttributeSet
import androidx.compose.foundation.layout.Arrangement
import androidx.compose.foundation.layout.fillMaxSize
import androidx.compose.foundation.lazy.LazyColumn
import androidx.compose.foundation.lazy.itemsIndexed
import androidx.compose.runtime.Composable
import androidx.compose.runtime.DisposableEffect
import androidx.compose.runtime.LaunchedEffect
import androidx.compose.runtime.collectAsState
import androidx.compose.runtime.getValue
import androidx.compose.ui.Modifier
import androidx.compose.ui.platform.AbstractComposeView
import androidx.compose.ui.platform.LocalLifecycleOwner
import androidx.compose.ui.unit.dp
import com.miam.core.localisation.Localisation
import com.miam.core.model.Recipe
import com.miam.core.viewModels.favoritePage.FavoritePageContract
import com.miam.core.viewModels.favoritePage.FavoritePageViewModel
import com.miam.kmm_miam_sdk.android.ressource.Image
import com.miam.sdk.components.baseComponent.ManagementResourceState
import com.miam.sdk.components.baseComponent.emptyPage.EmptyPageParameters
import com.miam.sdk.components.baseComponent.lazyListLoader.LazyListLoaderParameters
import com.miam.sdk.components.baseComponent.loader.LoaderParameters
import com.miam.sdk.components.recipeJourney.RecipeJourney
import com.miam.sdk.di.TemplateDI

class FavoritePage @JvmOverloads constructor(
    context: Context,
    attrs: AttributeSet? = null,
    defStyleAttr: Int = 0
): AbstractComposeView(context, attrs, defStyleAttr) {

    private var vmFavoritePage: FavoritePageViewModel = FavoritePageViewModel()

    fun bind(goVisitCatalog: () -> Unit) {
        this.goVisitCatalog = goVisitCatalog
    }

    private var goVisitCatalog = {}

    @Composable
    override fun Content() {
        DisposableEffect(Unit) { onDispose { vmFavoritePage.dispose() } }
        val state by vmFavoritePage.uiState.collectAsState()

        ManagementResourceState(
            resourceState = state.favoritesRecipes,
            successView = { favoritesRecipes ->
                requireNotNull(favoritesRecipes)
                FavoriteList(favoritesRecipes, context, state.isFetchingNewPage, !state.noMoreData) {
                    vmFavoritePage.setEvent(FavoritePageContract.Event.LoadPage)
                }
            },
            emptyView = {
                val params = EmptyPageParameters(
                    title = Localisation.Favorites.noFavorites.localised,
                    subtitle = "",
                    haveAnAction = true,
                    image = Image.favoriteFilled,
                    actionText = Localisation.Favorites.goToCatalog.localised,
                ) { goVisitCatalog() }

                TemplateDI.favoritePage.empty?.view?.Content(params = params)
                    ?: TemplateDI.defaultViews.empty.view.Content(params = params)
            },
            loadingView = {
                TemplateDI.favoritePage.loading?.view?.Content(params = LoaderParameters) ?: run {
                    TemplateDI.defaultViews.loading.view.Content(params = LoaderParameters)
                }
            }
        )
    }


    @Composable
    private fun FavoriteList(
        favoritesRecipes: List<Recipe>,
        context: Context,
        isLoading: Boolean,
        canLoadMore: Boolean,
        loadMore: () -> Unit
    ) {
        LazyColumn(
            modifier = Modifier.fillMaxSize(),
            verticalArrangement = Arrangement.spacedBy(TemplateDI.favoritePage.spacingBetweenCard.dp),
        ) {
            itemsIndexed(favoritesRecipes, { _, recipe -> recipe.id }) { index, recipe ->
                RecipeJourney(context = context).apply { bind(recipe = recipe) }.Content()
                if (index == favoritesRecipes.lastIndex && canLoadMore) {
                    LaunchedEffect(LocalLifecycleOwner.current) {
                        loadMore()
                    }
                    TemplateDI.favoritePage.lazyListLoader.view.Content(params = LazyListLoaderParameters(isLoading))
                }
            }
        }
    }
}