package com.miam.sdk.components.mealPlanner.basketPreview


import android.content.Context
import android.util.AttributeSet
import androidx.compose.foundation.layout.Box
import androidx.compose.foundation.layout.Column
import androidx.compose.foundation.layout.padding
import androidx.compose.material.Scaffold
import androidx.compose.runtime.Composable
import androidx.compose.runtime.DisposableEffect
import androidx.compose.runtime.LaunchedEffect
import androidx.compose.runtime.collectAsState
import androidx.compose.runtime.getValue
import androidx.compose.ui.Modifier
import androidx.compose.ui.platform.AbstractComposeView
import com.miam.core.base.state.ComponentUiState
import com.miam.core.viewModels.mealPlanner.basketPreview.MealPlannerBasketPreviewViewModel
import com.miam.sdk.components.baseComponent.loader.LoaderParameters
import com.miam.sdk.components.mealPlanner.basketPreview.empty.MealPlannerBasketPreviewEmptyParameters
import com.miam.sdk.components.mealPlanner.basketPreview.footer.MealPlannerBasketPreviewFooterParameters
import com.miam.sdk.components.mealPlanner.basketPreview.success.MealPlannerBasketPreviewSuccess
import com.miam.sdk.di.TemplateDI


class MealPlannerBasketPreview @JvmOverloads constructor(
    context: Context,
    attrs: AttributeSet? = null,
    defStyleAttr: Int = 0
): AbstractComposeView(context, attrs, defStyleAttr) {

    private val mealPlannerBasketPreviewViewModel = MealPlannerBasketPreviewViewModel()


    var onEmptyState: () -> Unit = {}
    var onComfirmBasket: () -> Unit = {}

    @Composable
    override fun Content() {

        DisposableEffect(mealPlannerBasketPreviewViewModel.hashCode()) {
            onDispose { mealPlannerBasketPreviewViewModel.dispose() }
        }

        LaunchedEffect(mealPlannerBasketPreviewViewModel.hashCode()) {
            mealPlannerBasketPreviewViewModel.registerListeners()
        }

        val state by mealPlannerBasketPreviewViewModel.uiState.collectAsState()
        val totalPrice by mealPlannerBasketPreviewViewModel.totalPrice.collectAsState()
        val productCount by mealPlannerBasketPreviewViewModel.productCount.collectAsState()

        Column {

            when (state.status) {
                ComponentUiState.LOADING -> {
                    TemplateDI.mealPlanner.basketPreview.loading?.view?.Content(LoaderParameters)
                        ?: TemplateDI.defaultViews.loading.view.Content(LoaderParameters)
                }

                ComponentUiState.SUCCESS, ComponentUiState.LOCKED -> {
                    Scaffold(bottomBar = {
                        BottomBar(totalPrice, productCount)
                    }
                    ) {
                        Box(modifier = Modifier.padding(it)) {
                            MealPlannerBasketPreviewSuccess(
                                mealPlannerBasketPreviewViewModel.recipesDetailsVM,
                            )
                        }
                    }
                }

                ComponentUiState.EMPTY -> {
                    TemplateDI.mealPlanner.basketPreview.empty.view.Content(mealPlannerBasketPreviewEmptyParameters = MealPlannerBasketPreviewEmptyParameters {
                        onEmptyState()
                    })
                }

                else -> {
                    // ERROR AND IDLE
                    Box {}
                }
            }
        }
    }

    @Composable
    fun BottomBar(totalPrice: Double, productCount: Int) {

        val isProcessingState by mealPlannerBasketPreviewViewModel.isProcessing.collectAsState()

        TemplateDI.mealPlanner.basketPreview.footer.view.Content(
            params =
            MealPlannerBasketPreviewFooterParameters(
                budget = mealPlannerBasketPreviewViewModel.budget,
                totalPrice = totalPrice,
                isProcessing = isProcessingState,
                productCount = productCount,
                confirmAction = {
                    mealPlannerBasketPreviewViewModel.onClickFinalize()
                    onComfirmBasket()
                },
                goToBasket = {}
            )
        )
    }
}