package com.miam.sdk.components.mealPlanner.meals.footer

import androidx.compose.animation.AnimatedVisibility
import androidx.compose.animation.core.FastOutSlowInEasing
import androidx.compose.animation.core.animateFloatAsState
import androidx.compose.animation.core.tween
import androidx.compose.animation.slideInVertically
import androidx.compose.animation.slideOutVertically
import androidx.compose.foundation.layout.Column
import androidx.compose.foundation.layout.Row
import androidx.compose.foundation.layout.Spacer
import androidx.compose.foundation.layout.fillMaxWidth
import androidx.compose.foundation.layout.height
import androidx.compose.foundation.layout.padding
import androidx.compose.foundation.layout.size
import androidx.compose.foundation.layout.width
import androidx.compose.foundation.shape.RoundedCornerShape
import androidx.compose.material.Button
import androidx.compose.material.ButtonDefaults
import androidx.compose.material.Icon
import androidx.compose.material.LinearProgressIndicator
import androidx.compose.material.Text
import androidx.compose.runtime.Composable
import androidx.compose.ui.Alignment
import androidx.compose.ui.Modifier
import androidx.compose.ui.draw.clip
import androidx.compose.ui.graphics.Color
import androidx.compose.ui.graphics.RectangleShape
import androidx.compose.ui.res.painterResource
import androidx.compose.ui.unit.dp
import com.miam.core.localisation.Localisation
import java.text.NumberFormat
import java.util.Currency

internal class MealPlannerFooterImp: MealPlannerFooter {
    @Composable
    override fun Content(params: MealsFooterParameters) {

        var ratio = params.recipesPrice / params.budget
        val isBudgetExceeded =
            if (ratio > 1) com.miam.kmm_miam_sdk.android.theme.Colors.dangerBudgetText else com.miam.kmm_miam_sdk.android.theme.Colors.positive
        val numberFormat = NumberFormat.getCurrencyInstance()
        numberFormat.currency = Currency.getInstance(Localisation.Price.currency.localised)

        val progressAnimation = animateFloatAsState(
            targetValue = if (ratio > 1) (params.budget / params.recipesPrice).toFloat()
            else (ratio).toFloat(),
            animationSpec = tween(durationMillis = 1500, easing = FastOutSlowInEasing)
        ).value

        AnimatedVisibility(
            visible = params.recipesPrice != 0.0,
            enter = slideInVertically { height -> height },
            exit = slideOutVertically { height -> height }
        ) {
            Column {

                Row(
                    modifier = Modifier
                        .fillMaxWidth()
                        .height(45.dp),
                    verticalAlignment = Alignment.CenterVertically
                ) {
                    if (ratio > 1) {
                        LinearProgressIndicator(
                            backgroundColor = com.miam.kmm_miam_sdk.android.theme.Colors.dangerBudgetBar,
                            progress = progressAnimation,
                            modifier = Modifier
                                .height(8.dp)
                                .padding(horizontal = 32.dp)
                                .clip(RoundedCornerShape(8.dp)),
                            color = com.miam.kmm_miam_sdk.android.theme.Colors.primary
                        )
                    } else {
                        LinearProgressIndicator(
                            progress = progressAnimation,
                            modifier = Modifier
                                .height(8.dp)
                                .padding(horizontal = 32.dp)
                                .clip(RoundedCornerShape(8.dp)),
                            color = com.miam.kmm_miam_sdk.android.theme.Colors.primary
                        )
                    }
                    Text(
                        text = numberFormat.format(params.recipesPrice),
                        color = isBudgetExceeded,
                        style = com.miam.kmm_miam_sdk.android.theme.Typography.bodyBold
                    )
                }


                Button(
                    onClick = { params.addAllToBasket() },
                    modifier = Modifier
                        .fillMaxWidth()
                        .height(45.dp),
                    colors = ButtonDefaults.buttonColors(backgroundColor = com.miam.kmm_miam_sdk.android.theme.Colors.primary),
                    shape = RectangleShape,
                ) {
                    Icon(
                        painterResource(id = com.miam.kmm_miam_sdk.android.ressource.Image.cart),
                        contentDescription = null,
                        tint = com.miam.kmm_miam_sdk.android.theme.Colors.white,
                        modifier = Modifier.size(20.dp)
                    )
                    Spacer(modifier = Modifier.width(8.dp))
                    Text(
                        text = Localisation.Recipe.add.localised,
                        color = Color.White,
                        style = com.miam.kmm_miam_sdk.android.theme.Typography.body
                    )
                }
            }
        }
    }
}