package com.miam.sdk.components.preferences

import android.content.Context
import android.util.AttributeSet
import androidx.compose.foundation.background
import androidx.compose.foundation.layout.Column
import androidx.compose.foundation.layout.Spacer
import androidx.compose.foundation.layout.fillMaxWidth
import androidx.compose.foundation.layout.height
import androidx.compose.foundation.layout.padding
import androidx.compose.foundation.rememberScrollState
import androidx.compose.foundation.verticalScroll
import androidx.compose.material.Scaffold
import androidx.compose.runtime.Composable
import androidx.compose.runtime.collectAsState
import androidx.compose.runtime.getValue
import androidx.compose.ui.Modifier
import androidx.compose.ui.platform.AbstractComposeView
import androidx.compose.ui.platform.LocalContext
import androidx.compose.ui.unit.dp
import com.miam.core.di.MiamDI
import com.miam.core.handler.LogHandler
import com.miam.core.localisation.Localisation
import com.miam.core.viewModels.preferences.PreferencesContent
import com.miam.core.viewModels.preferences.SingletonPreferencesViewModel
import com.miam.kmm_miam_sdk.android.theme.Colors
import com.miam.sdk.components.baseComponent.ManagementResourceState
import com.miam.sdk.components.baseComponent.emptyPage.EmptyPageParameters
import com.miam.sdk.components.baseComponent.loader.LoaderParameters
import com.miam.sdk.components.preferences.search.PreferencesSearch
import com.miam.sdk.components.preferences.success.dietSection.PreferencesDietSectionParameters
import com.miam.sdk.components.preferences.success.equipmentSection.PreferencesEquipmentSectionParameters
import com.miam.sdk.components.preferences.success.footer.PreferencesFooterParameters
import com.miam.sdk.components.preferences.success.guestSection.PreferencesGuestSectionParameters
import com.miam.sdk.components.preferences.success.header.PreferencesSuccessHeaderParameters
import com.miam.sdk.components.preferences.success.ingredientSection.PreferencesIngredientSectionParameters
import com.miam.sdk.di.TemplateDI

class Preferences @JvmOverloads constructor(
    context: Context,
    attrs: AttributeSet? = null,
    defStyleAttr: Int = 0
): AbstractComposeView(context, attrs, defStyleAttr) {

    private var close: () -> Unit = {
        LogHandler.error("[Miam][Error] you must bind close function")
    }

    private var onApply: () -> Unit = {
        LogHandler.error("[Miam][Error] you must bind close function")
    }

    fun bind(close: () -> Unit, onApply: () -> Unit) {
        this.close = close
        this.onApply = onApply
    }

    private val preferencesVM: SingletonPreferencesViewModel by lazy { MiamDI.preferencesViewModel }

    private fun resetAndClose() {
        preferencesVM.resetPreferences()
        close()
    }

    private fun applyAndClose() {
        preferencesVM.applyPreferences()
        onApply()
    }

    private fun back() {
        preferencesVM.back()
    }

    @Composable
    override fun Content() {
        val state by preferencesVM.uiState.collectAsState()

        ManagementResourceState(
            resourceState = state.basicState,
            successView = { content ->
                requireNotNull(content)
                when (content) {

                    PreferencesContent.SEARCH_PREFRERENCES -> {
                        PreferencesSearch(LocalContext.current).apply {
                            bind(::back) {
                                preferencesVM.addIngredientPreference(it)
                                back()
                            }
                        }.Content()
                    }

                    PreferencesContent.ALL_PREFRENCES -> {
                        Scaffold(
                            topBar = {
                                TemplateDI.preferences.success.header.view.Content(params = PreferencesSuccessHeaderParameters {
                                    resetAndClose()
                                })
                            },
                            content = { padding ->
                                Column(
                                    Modifier
                                        .fillMaxWidth()
                                        .background(color = Colors.backgroundGrey)
                                        .padding(padding)
                                        .padding(horizontal = 16.dp)
                                        .verticalScroll(rememberScrollState())
                                ) {
                                    TemplateDI.preferences.success.guestSection.view.Content(
                                        params = PreferencesGuestSectionParameters(
                                            guests = state.guests
                                        ) { preferencesVM.changeGlobalGuest(it) }
                                    )
                                    Spacer(modifier = Modifier.height(24.dp))
                                    TemplateDI.preferences.success.dietSection.view.Content(
                                        params = PreferencesDietSectionParameters(
                                            dietsTag = state.diets
                                        ) { preferencesVM.togglePreference(it) }
                                    )
                                    Spacer(modifier = Modifier.height(24.dp))
                                    TemplateDI.preferences.success.ingredientSection.view.Content(
                                        params = PreferencesIngredientSectionParameters(
                                            ingredientsTag = state.ingredients,
                                            togglePreference = { preferencesVM.togglePreference(it) },
                                        ) { preferencesVM.goToSearchPrefAndPushRoute() }
                                    )
                                    Spacer(modifier = Modifier.height(24.dp))
                                    TemplateDI.preferences.success.equipmentSection.view.Content(
                                        params = PreferencesEquipmentSectionParameters(state.equipments) {
                                            preferencesVM.togglePreference(it)
                                        }
                                    )
                                    Spacer(modifier = Modifier.height(100.dp))
                                }
                            },
                            bottomBar = {
                                TemplateDI.preferences.success.footer.view.Content(
                                    params = PreferencesFooterParameters(
                                        recipesFound = state.recipesFound,
                                        closePref = ::resetAndClose,
                                        applyPref = ::applyAndClose
                                    )
                                )
                            }
                        )
                    }
                }
            },
            loadingView = {
                TemplateDI.preferences.loader?.view?.Content(params = LoaderParameters)
                    ?: TemplateDI.defaultViews.loading.view.Content(params = LoaderParameters)
            },
            emptyView = {
                val params = EmptyPageParameters(title = Localisation.Preferences.emptyTitle.localised)
                TemplateDI.preferences.empty?.view?.Content(params = params)
                    ?: TemplateDI.defaultViews.empty.view.Content(params = params)
            }
        )
    }
}