package com.miam.sdk.components.mealPlanner.meals.recipe.success

import androidx.compose.foundation.Canvas
import androidx.compose.foundation.Image
import androidx.compose.foundation.background
import androidx.compose.foundation.clickable
import androidx.compose.foundation.gestures.detectHorizontalDragGestures
import androidx.compose.foundation.layout.Arrangement
import androidx.compose.foundation.layout.Box
import androidx.compose.foundation.layout.Column
import androidx.compose.foundation.layout.Row
import androidx.compose.foundation.layout.Spacer
import androidx.compose.foundation.layout.fillMaxSize
import androidx.compose.foundation.layout.fillMaxWidth
import androidx.compose.foundation.layout.height
import androidx.compose.foundation.layout.offset
import androidx.compose.foundation.layout.padding
import androidx.compose.foundation.layout.size
import androidx.compose.foundation.layout.width
import androidx.compose.foundation.shape.RoundedCornerShape
import androidx.compose.material.Button
import androidx.compose.material.ButtonDefaults
import androidx.compose.material.Card
import androidx.compose.material.Surface
import androidx.compose.material.Text
import androidx.compose.material.TextButton
import androidx.compose.runtime.Composable
import androidx.compose.runtime.getValue
import androidx.compose.runtime.mutableStateOf
import androidx.compose.runtime.remember
import androidx.compose.runtime.setValue
import androidx.compose.ui.Alignment
import androidx.compose.ui.Modifier
import androidx.compose.ui.draw.clip
import androidx.compose.ui.geometry.Offset
import androidx.compose.ui.graphics.Color
import androidx.compose.ui.graphics.ColorFilter
import androidx.compose.ui.input.pointer.pointerInput
import androidx.compose.ui.layout.ContentScale
import androidx.compose.ui.res.painterResource
import androidx.compose.ui.text.font.FontWeight
import androidx.compose.ui.text.style.TextAlign
import androidx.compose.ui.text.style.TextOverflow
import androidx.compose.ui.tooling.preview.Preview
import androidx.compose.ui.unit.Dp
import androidx.compose.ui.unit.IntOffset
import androidx.compose.ui.unit.dp
import coil.compose.AsyncImage
import com.miam.core.localisation.Localisation
import com.miam.core.model.RecipeAttributes
import com.miam.core.model.fake.RecipeFakeFactory
import com.miam.kmm_miam_sdk.android.ressource.Image.delete
import com.miam.kmm_miam_sdk.android.ressource.Image.miamDifficulty
import com.miam.kmm_miam_sdk.android.ressource.Image.miamTime
import com.miam.kmm_miam_sdk.android.theme.Colors
import com.miam.kmm_miam_sdk.android.theme.Colors.primary
import com.miam.kmm_miam_sdk.android.theme.Dimension
import com.miam.kmm_miam_sdk.android.theme.Typography
import com.miam.sdk.components.baseComponent.likeButton.LikeButton
import com.miam.sdk.components.recipeCard.success.PricePerPerson
import kotlin.math.roundToInt

class MealPlannerRecipeCardSuccessImp: MealPlannerRecipeSuccessCard {
    @Composable
    override fun Content(mealPlannerRecipeCardParameters: MealPlannerRecipeCardSuccessParameters) {
        if (mealPlannerRecipeCardParameters.isInSearchPage) {
            return RecipeCardSearchPageContent(params = mealPlannerRecipeCardParameters)
        }
        RecipeCardContent(params = mealPlannerRecipeCardParameters)
    }

    @Preview
    @Composable
    fun MealPlannerRecipeCardImpPreview() {
        MealPlannerRecipeCardSuccessImp().Content(mealPlannerRecipeCardParameters = MealPlannerRecipeCardSuccessParameters(
            index = 1,
            isInSearchPage = false,
            isAlreadyInBasket = false,
            guest = 4,
            price = 45.0,
            recipe = RecipeFakeFactory.create(attributes = RecipeAttributes(title = "Deep Dish", mediaUrl = "https://picsum.photos/200/300")),
            changeAction = {},
            replaceAction = { },
            deleteAction = {},
            openDetail = {}
        ))
    }
}

@Composable
fun RecipeCardContent(params: MealPlannerRecipeCardSuccessParameters) {
    var offsetX by remember { mutableStateOf(0f) }
    val threshold = 200f
    val maxSwipeDistance = 250f // Change this value to adjust the maximum swipe distance
    val showCloseIcon = remember { mutableStateOf(false) }

    Box(Modifier.padding(vertical = 8.dp)) {
        Button(
            onClick = {
                params.deleteAction()
            },
            colors = ButtonDefaults.buttonColors(
                backgroundColor = Colors.miamDangerBackground
            ),
            modifier = Modifier
                .align(Alignment.CenterEnd)
                .height(Dimension.mealPlannerCardHeight)
                .width(maxSwipeDistance.dp)
        ) {
            Row(
                modifier = Modifier.fillMaxSize(),
                horizontalArrangement = Arrangement.End,
                verticalAlignment = Alignment.CenterVertically
            ) {
                Image(
                    painter = painterResource(delete),
                    contentDescription = null,
                    colorFilter = ColorFilter.tint(Colors.miamDangerText),
                    modifier = Modifier.size(30.dp)
                )
            }
        }
        Box(
            modifier = Modifier
                .height(Dimension.mealPlannerCardHeight)
                .offset { IntOffset(offsetX.roundToInt(), 0) }
                .pointerInput(Unit) {
                    detectHorizontalDragGestures { change, dragAmount ->
                        val newOffsetX = offsetX + dragAmount
                        if (newOffsetX <= 0f && newOffsetX >= -maxSwipeDistance) {
                            offsetX = newOffsetX
                        }
                        change.consume()
                        showCloseIcon.value = offsetX <= -threshold
                    }
                }
        ) {
            RecipeCardFace(params)
        }
    }
}

@Composable
fun RecipeCardFace(params: MealPlannerRecipeCardSuccessParameters) {
    Box(
        modifier = Modifier
            .height(Dimension.mealPlannerCardHeight)
            .background(color = Color.White)
    ) {
        Canvas(modifier = Modifier.matchParentSize()) {
            val strokeWidthPx = Dimension.borderWidth.toPx()
            val yTop = strokeWidthPx / 4
            val yBottom = size.height - strokeWidthPx / 4

            drawLine(
                color = Color.LightGray,
                Offset(x = 0f, y = yTop),
                Offset(size.width, yTop),
                strokeWidthPx
            )
            drawLine(
                color = Color.LightGray,
                Offset(x = 0f, y = yBottom),
                Offset(size.width, yBottom),
                strokeWidthPx
            )
        }
        Row(
            verticalAlignment = Alignment.CenterVertically,
        ) {
            Box {
                AsyncImage(
                    model = params.recipe.attributes!!.mediaUrl,
                    contentDescription = "Grocery Item Image",
                    modifier = Modifier
                        .height(Dimension.mealPlannerCardHeight)
                        .width(150.dp)
                        .clickable { params.openDetail() },
                    contentScale = ContentScale.Crop
                )
                BudgetLikeButton(params.recipe.id, 150.dp)
            }
            Column(
                modifier = Modifier
                    .fillMaxSize()
                    .padding(Dimension.lPadding),
                verticalArrangement = Arrangement.SpaceBetween,
                horizontalAlignment = Alignment.CenterHorizontally,
            ) {
                Text(
                    text = params.recipe.attributes!!.title,
                    style = Typography.link,
                    textAlign = TextAlign.Center,
                    maxLines = 2,
                    overflow = TextOverflow.Ellipsis,
                    modifier = Modifier.padding(horizontal = Dimension.lPadding)
                )
                Row(
                    Modifier.fillMaxWidth(),
                    verticalAlignment = Alignment.CenterVertically,
                    horizontalArrangement = Arrangement.Center
                ) {
                    when (params.recipe.attributes!!.difficulty) {
                        1 -> BudgetMetricView(Localisation.Recipe.lowDifficulty.localised, miamDifficulty)
                        2 -> BudgetMetricView(Localisation.Recipe.mediumDifficulty.localised, miamDifficulty)
                        3 -> BudgetMetricView(Localisation.Recipe.highDifficulty.localised, miamDifficulty)
                    }
                    BudgetMetricView(params.recipe.totalTime, miamTime)

                }
                TextButton(
                    onClick = { params.changeAction() },
                    colors = ButtonDefaults.textButtonColors(contentColor = Colors.primary)
                ) {
                    Text(text = Localisation.Budget.myBudgetChangeRecipe.localised, style = Typography.link)
                }
            }
        }
    }
}

@Composable
fun RecipeCardSearchPageContent(params: MealPlannerRecipeCardSuccessParameters) {

    Card(
        Modifier.fillMaxWidth()
    ) {
        Column(horizontalAlignment = Alignment.CenterHorizontally) {
            Box {
                params.recipe.attributes?.mediaUrl?.let { imageUrl ->
                    AsyncImage(
                        model = imageUrl,
                        contentDescription = "Grocery Item Image",
                        modifier = Modifier
                            .height(Dimension.mealPlannerCardHeight)
                            .fillMaxWidth()
                            .clickable { params.openDetail() },
                        contentScale = ContentScale.Crop
                    )
                }
                BudgetLikeButton(params.recipe.id)
            }
            Spacer(modifier = Modifier.height(12.dp))
            Text(
                text = params.recipe.attributes!!.title,
                style = Typography.link,
                textAlign = TextAlign.Center,
                maxLines = 2,
                overflow = TextOverflow.Ellipsis,
                modifier = Modifier
                    .padding(horizontal = Dimension.lPadding)
                    .height(40.dp)
            )
            Spacer(modifier = Modifier.height(12.dp))
            Row(
                Modifier
                    .fillMaxWidth()
                    .padding(horizontal = 12.dp),
                horizontalArrangement = Arrangement.SpaceBetween,
                verticalAlignment = Alignment.CenterVertically
            ) {
                Box(modifier = Modifier.width(70.dp)) {
                    PricePerPerson(params.recipe.attributes?.price?.pricePerServe ?: 0.0)
                }
                Box(
                    modifier = Modifier.weight(1f)
                ) {
                    MealPlannerRecipePickerCardCTAView(params.replaceAction)
                }
            }
            Spacer(modifier = Modifier.height(12.dp))
        }
    }
}

@Composable
fun MealPlannerRecipePickerCardCTAView(
    actionOnClick: () -> Unit
) {
    Row(Modifier.fillMaxWidth(), horizontalArrangement = Arrangement.End) {
        Surface(
            modifier = Modifier
                .clip(RoundedCornerShape(8.dp))
                .clickable {
                    actionOnClick()
                },
            elevation = 8.dp
        ) {
            Row(
                Modifier
                    .background(primary)
                    .padding(horizontal = 12.dp, vertical = 8.dp),
                verticalAlignment = Alignment.CenterVertically,
                horizontalArrangement = Arrangement.spacedBy(8.dp)
            ) {
                Image(
                    painter = painterResource(com.miam.kmm_miam_sdk.android.ressource.Image.cart),
                    contentDescription = "recipe is in cart icon",
                    colorFilter = ColorFilter.tint(Colors.white),
                    modifier = Modifier
                        .size(24.dp)
                )
            }
        }
    }
}

@Composable
fun BudgetLikeButton(recipeId: String, sizeOfImage: Dp? = null) {
    Row(
        modifier = Modifier
            .then(
                if (sizeOfImage != null) Modifier.width(sizeOfImage)
                else Modifier.fillMaxWidth()
            )
            .padding(top = Dimension.mPadding, end = Dimension.mPadding),
        horizontalArrangement = Arrangement.End
    ) {
        LikeButton(recipeId = recipeId).Content()
    }
}

@Composable
fun BudgetMetricView(text: String, image: Int) {
    Column(
        horizontalAlignment = Alignment.CenterHorizontally,
        verticalArrangement = Arrangement.Top,
        modifier = Modifier
            .padding(horizontal = 16.dp)
    ) {
        Image(
            painter = painterResource(image),
            contentDescription = null,
            colorFilter = ColorFilter.tint(Colors.boldText),
            modifier = Modifier.size(22.dp)
        )
        Spacer(modifier = Modifier.height(4.dp))
        Text(
            text = text,
            color = Colors.boldText,
            style = Typography.overLine.copy(fontWeight = FontWeight.Bold),
            textAlign = TextAlign.Center,
            modifier = Modifier.padding(top = 8.dp)
        )
    }
}