package com.miam.sdk.components.baseComponent.recipeCard

import android.content.Context
import android.util.AttributeSet
import androidx.compose.animation.Crossfade
import androidx.compose.animation.core.tween
import androidx.compose.foundation.layout.Column
import androidx.compose.runtime.Composable
import androidx.compose.runtime.DisposableEffect
import androidx.compose.runtime.collectAsState
import androidx.compose.runtime.getValue
import androidx.compose.ui.platform.AbstractComposeView
import com.miam.core.model.Recipe
import com.miam.core.viewModels.recipe.RecipeContract
import com.miam.core.viewModels.recipe.RecipeViewModel
import com.miam.sdk.components.baseComponent.ManagementResourceState

abstract class BaseRecipeCard @JvmOverloads constructor(
    context: Context,
    attrs: AttributeSet? = null,
    defStyleAttr: Int = 0
): AbstractComposeView(context, attrs, defStyleAttr) {

    internal open val vmRecipe: RecipeViewModel = RecipeViewModel()
    internal var idRecipeState: String? = null

    var idRecipe: String
        get() = idRecipeState ?: ""
        set(value) {
            idRecipeState = value
            vmRecipe.setEvent(RecipeContract.Event.FetchRecipeFromId(value))
        }

    var recipe: Recipe? = null
        set(value) {
            field = value
            value?.let {
                vmRecipe.setEvent(RecipeContract.Event.FetchRecipeFromId(it.id))
            }
        }

    @Composable
    override fun Content() {
        DisposableEffect(Unit) { onDispose { vmRecipe.dispose() } }
        Column {
            UpdatableContent()
        }
    }

    @Composable
    private fun UpdatableContent() {
        val state by vmRecipe.uiState.collectAsState()
        Crossfade(targetState = state.recipeState, animationSpec = tween(750, 0), label = "transition animation") { recipeState ->
            ManagementResourceState(
                resourceState = recipeState,
                successView = { recipe ->
                    requireNotNull(recipe)
                    SuccessView(recipe = recipe)
                },
                loadingView = {
                    LoadingView()
                },
                emptyView = {
                    EmptyView()
                }
            )
        }
    }

    @Composable
    abstract fun SuccessView(recipe: Recipe)

    @Composable
    abstract fun LoadingView()

    @Composable
    abstract fun EmptyView()

}
