package com.miam.sdk.components.baseComponent.filter.success

import androidx.compose.foundation.BorderStroke
import androidx.compose.foundation.Image
import androidx.compose.foundation.background
import androidx.compose.foundation.border
import androidx.compose.foundation.layout.Arrangement
import androidx.compose.foundation.layout.Box
import androidx.compose.foundation.layout.Column
import androidx.compose.foundation.layout.Row
import androidx.compose.foundation.layout.fillMaxSize
import androidx.compose.foundation.layout.fillMaxWidth
import androidx.compose.foundation.layout.padding
import androidx.compose.foundation.rememberScrollState
import androidx.compose.foundation.shape.RoundedCornerShape
import androidx.compose.foundation.verticalScroll
import androidx.compose.material.Checkbox
import androidx.compose.material.CheckboxDefaults
import androidx.compose.material.Divider
import androidx.compose.material.Text
import androidx.compose.runtime.Composable
import androidx.compose.runtime.MutableState
import androidx.compose.runtime.mutableStateOf
import androidx.compose.ui.Alignment
import androidx.compose.ui.Modifier
import androidx.compose.ui.draw.clip
import androidx.compose.ui.res.painterResource
import androidx.compose.ui.unit.dp
import com.miam.core.localisation.Localisation
import com.miam.core.model.CatalogFilterOptions
import com.miam.kmm_miam_sdk.android.ressource.Image
import com.miam.kmm_miam_sdk.android.theme.Colors
import com.miam.kmm_miam_sdk.android.theme.Typography
import com.miam.kmm_miam_sdk.android.ui.components.common.Clickable

internal class FilterSuccessImp: FilterSuccess {
    @Composable
    override fun Content(params: FilterSuccessParameters) {
        Box(
            modifier = Modifier
                .fillMaxSize()
                .background(Colors.white)
        ) {
            Column(
                Modifier.padding(vertical = 8.dp, horizontal = 16.dp),
                horizontalAlignment = Alignment.CenterHorizontally
            ) {
                FilterHeader(closeDialog = params.closeDialog)
                Column(
                    Modifier
                        .weight(weight = 1f, fill = false)
                        .verticalScroll(rememberScrollState())
                ) {
                    FilterSection(
                        text = Localisation.Catalog.difficulty.localised,
                        catalogFilterOptions = params.difficulties,
                        params.onDifficultyChanged
                    )
                    Divider(Modifier.padding(vertical = 16.dp))
                    FilterSection(
                        text = Localisation.Catalog.costPerPerson.localised,
                        catalogFilterOptions = params.costs,
                        params.onCostFilterChanged
                    )
                    Divider(Modifier.padding(vertical = 16.dp))
                    FilterSection(
                        text = Localisation.Catalog.preparationTime.localised,
                        catalogFilterOptions = params.times,
                        params.onTimeFilterChanged
                    )
                }
                ClearButton(params.clearFilter)
                Divider(Modifier.padding(vertical = 8.dp))
                ApplyAndGoButton(params.applyAndGo, params.numberOfResult)
            }
        }
    }

    @Composable
    fun FilterHeader(closeDialog: () -> Unit) {
        Row(Modifier.fillMaxWidth(), horizontalArrangement = Arrangement.SpaceBetween) {
            Text(Localisation.Catalog.filtersTitle.localised, color = Colors.black, style = Typography.subtitleBold)
            Clickable(
                onClick = closeDialog,
                children = { Image(painter = painterResource(Image.close), contentDescription = "Close Filters") }
            )
        }
    }

    @Composable
    fun FilterSection(text: String, catalogFilterOptions: List<CatalogFilterOptions>, onCheckedCallback: (CatalogFilterOptions) -> Unit) {
        Text(text = text, style = Typography.bodyBold)
        catalogFilterOptions.forEach { catOption ->
            CheckboxRow(catOption, mutableStateOf(catOption.isSelected)) { onCheckedCallback(catOption) }
        }
    }

    @Composable
    fun CheckboxRow(
        catOption: CatalogFilterOptions,
        checkedState: MutableState<Boolean>,
        updateFilter: (catOption: CatalogFilterOptions) -> Unit
    ) {
        Row(verticalAlignment = Alignment.CenterVertically) {
            Checkbox(
                checked = checkedState.value,
                onCheckedChange = {
                    checkedState.value = it
                    updateFilter(catOption)
                },
                colors = CheckboxDefaults.colors(Colors.primary)
            )
            Text(text = catOption.uiLabel)
        }
    }

    @Composable
    fun ClearButton(clearFilter: () -> Unit) {
        Clickable(onClick = { clearFilter() }) {
            Box(
                modifier = Modifier.border(border = BorderStroke(1.dp, Colors.primary), shape = RoundedCornerShape(50))
            ) {
                Text(
                    text = Localisation.Catalog.removeFilters.localised,
                    color = Colors.primary,
                    modifier = Modifier.padding(horizontal = 16.dp, vertical = 8.dp)
                )
            }
        }
    }

    @Composable
    fun ApplyAndGoButton(applyAndGo: () -> Unit, numberOfResult: Int) {
        Clickable(onClick = applyAndGo) {
            Box(
                Modifier
                    .clip(RoundedCornerShape(50))
                    .background(Colors.primary)
            ) {
                Text(
                    text = Localisation.Catalog.showResults(numberOfResult).localised,
                    color = Colors.white,
                    modifier = Modifier.padding(horizontal = 16.dp, vertical = 8.dp)
                )
            }
        }
    }
}