package com.miam.sdk.components.mealPlanner.meals

import android.content.Context
import android.util.AttributeSet
import androidx.compose.runtime.Composable
import androidx.compose.runtime.collectAsState
import androidx.compose.runtime.getValue
import androidx.compose.runtime.rememberCoroutineScope
import androidx.compose.ui.platform.AbstractComposeView
import androidx.lifecycle.Lifecycle
import com.miam.core.base.state.ComponentUiState
import com.miam.core.handler.LogHandler
import com.miam.core.viewModels.mealPlanner.form.MealPlannerFormViewModel
import com.miam.sdk.components.baseComponent.lifeCycle.OnLifecycleEvent
import com.miam.sdk.components.mealPlanner.form.success.MealPlannerFormSuccessParameters
import com.miam.sdk.di.TemplateDI
import kotlinx.coroutines.CoroutineExceptionHandler
import kotlinx.coroutines.launch

class MealPlannerToolbar @JvmOverloads constructor(
    context: Context,
    attrs: AttributeSet? = null,
    defStyleAttr: Int = 0
): AbstractComposeView(context, attrs, defStyleAttr) {

    private val formViewModel = MealPlannerFormViewModel()
    private var onRecipeFounded: (recipesId: List<String>) -> Unit = {}

    fun bind(onRecipeFounded: (recipesId: List<String>) -> Unit = {}) {
        this.onRecipeFounded = onRecipeFounded
    }

    private val coroutineHandler = CoroutineExceptionHandler { _, exception ->
        LogHandler.error("Miam error in MealPlannerToolbar $exception")
    }

    private suspend fun loadRecipe(budget: Int, numberOfMeals: Int, numberOfGuests: Int) {
        try {
            val recipes = formViewModel.getRecipesForBudgetConstraint(budget, numberOfMeals, numberOfGuests)
            if (formViewModel.currentState.uiState != ComponentUiState.SUCCESS) return
            onRecipeFounded(recipes)
        } catch (e: Throwable) {
            formViewModel.getRecipesMaxCountForBudgetConstraint(budget, numberOfMeals)
        }
    }

    @Composable
    override fun Content() {
        val state by formViewModel.uiState.collectAsState()
        val scope = rememberCoroutineScope()

        OnLifecycleEvent { _, event ->
            when (event) {
                Lifecycle.Event.ON_CREATE -> {
                    formViewModel.getRecipesMaxCountForBudgetConstraint(state.budget, state.numberOfMeals)
                }

                else -> {
                    return@OnLifecycleEvent
                }
            }
        }

        TemplateDI.mealPlanner.toolbar.view.Content(
            params = MealPlannerFormSuccessParameters(
                budget = state.budget,
                numberOfGuests = state.numberOfGuests,
                numberOfMeals = state.numberOfMeals,
                uiState = state.uiState,
                setBudget = { formViewModel.setBudget(it) },
                setNumberOfGuests = { formViewModel.setNumberOfGuests(it) },
                setNumberOfMeals = { formViewModel.setNumberOfMeals(it) },
                maxMealCount = state.recipeMaxCount,
                currentRemainingRecipeCountFlow = formViewModel.recipeCountFlow,
                refreshMaxMealCount = { budget, numberOfGuests ->
                    formViewModel.getRecipesMaxCountForBudgetConstraint(budget, numberOfGuests)
                },
                submit = { budget, numberOfMeals, numberOfGuests ->
                    scope.launch(coroutineHandler) { loadRecipe(budget, numberOfMeals, numberOfGuests) }
                }
            )
        )
    }
}