package com.miam.sdk.components.mealPlanner.basketPreview.success.found

import androidx.compose.foundation.layout.Column
import androidx.compose.runtime.Composable
import androidx.compose.runtime.DisposableEffect
import androidx.compose.runtime.LaunchedEffect
import androidx.compose.runtime.collectAsState
import androidx.compose.runtime.getValue
import androidx.compose.runtime.mutableStateOf
import androidx.compose.runtime.remember
import androidx.compose.ui.ExperimentalComposeUiApi
import androidx.compose.ui.window.Dialog
import androidx.compose.ui.window.DialogProperties
import com.miam.core.base.state.ComponentUiState
import com.miam.core.viewModels.itemSelector.ItemSelectorContract
import com.miam.core.viewModels.itemSelector.ItemSelectorViewModel
import com.miam.core.viewModels.product.ProductContract
import com.miam.core.viewModels.product.ProductViewModel
import com.miam.sdk.components.itemSelector.ItemSelector
import com.miam.sdk.components.mealPlanner.basketPreview.success.found.products.FoundProductParameters
import com.miam.sdk.di.TemplateDI
import kotlinx.coroutines.flow.MutableStateFlow
import java.util.Locale

@OptIn(ExperimentalComposeUiApi::class)
@Composable
fun FoundProducts(foundProduct: MutableStateFlow<List<ProductViewModel>>) {

    val foundProduct by foundProduct.collectAsState()

    return Column {
        // TODO can be improve and become a grid
        Column {
            foundProduct.forEach { productVm ->

                val productState = productVm.uiState.collectAsState()
                val isDialogOpen = remember { mutableStateOf(false) }

                fun closeDialog() {
                    isDialogOpen.value = false
                }

                DisposableEffect(Unit) { onDispose { productVm.dispose() } }
                LaunchedEffect(Unit) { productVm.registerListeners() }

                val itemSelectorViewModel = remember { ItemSelectorViewModel() }

                if (isDialogOpen.value) {
                    Dialog(
                        onDismissRequest = ::closeDialog,
                        properties = DialogProperties(usePlatformDefaultWidth = false)
                    ) {
                        productState.value.ingredient?.let { ingredient ->
                            ItemSelector.View(itemSelectorViewModel, ingredient.id) {
                                itemSelectorViewModel.setEvent(ItemSelectorContract.Event.ReturnToDetail)
                                closeDialog()
                            }
                        }
                    }
                }
                productState.value.ingredient?.let { ingredient ->
                    productState.value.basketEntry?.let { basketEntry ->
                        basketEntry.relationships?.selectedItem?.data?.let { selectedItem ->
                            TemplateDI.mealPlanner.basketPreview.success.foundProduct.product.view.Content(
                                FoundProductParameters(
                                    ingredientId = ingredient.id,
                                    ingredientName = ingredient.name.replaceFirstChar { if (it.isLowerCase()) it.titlecase(Locale.ROOT) else it.toString() },
                                    productName = selectedItem.attributes?.name ?: "",
                                    productCapacityUnit = "${selectedItem.attributes?.capacityVolume}  ${selectedItem.attributes?.capacityUnit}",
                                    productPicture = selectedItem.attributes?.image ?: "",
                                    quantity = basketEntry.quantity,
                                    shareInRecipeCount = basketEntry.recipeCount,
                                    price = basketEntry.unitPrice,
                                    isReloading = productState.value.status == ComponentUiState.LOCKED,
                                    ean = selectedItem.attributes?.ean ?: "",
                                    delete = { productVm.setEvent(ProductContract.Event.IgnoreIngredient) },
                                    replace = {
                                        productVm.setEvent(ProductContract.Event.ReplaceProduct)
                                        isDialogOpen.value = true
                                    },
                                    onQuantityChanged = {
                                        productVm.setEvent(ProductContract.Event.UpdateProductQty(it))
                                    }
                                )
                            )
                        }
                    }
                }
            }
        }
    }
}