package com.miam.sdk.components.mealPlanner.basketPreview.success.found.products

import androidx.compose.foundation.BorderStroke
import androidx.compose.foundation.Image
import androidx.compose.foundation.background
import androidx.compose.foundation.border
import androidx.compose.foundation.layout.Arrangement
import androidx.compose.foundation.layout.Box
import androidx.compose.foundation.layout.Column
import androidx.compose.foundation.layout.Row
import androidx.compose.foundation.layout.Spacer
import androidx.compose.foundation.layout.fillMaxWidth
import androidx.compose.foundation.layout.height
import androidx.compose.foundation.layout.padding
import androidx.compose.foundation.layout.size
import androidx.compose.foundation.layout.width
import androidx.compose.foundation.layout.widthIn
import androidx.compose.foundation.shape.RoundedCornerShape
import androidx.compose.material.CircularProgressIndicator
import androidx.compose.material.Divider
import androidx.compose.material.IconButton
import androidx.compose.material.Surface
import androidx.compose.material.Text
import androidx.compose.runtime.Composable
import androidx.compose.runtime.getValue
import androidx.compose.runtime.mutableStateOf
import androidx.compose.runtime.remember
import androidx.compose.runtime.setValue
import androidx.compose.ui.Alignment
import androidx.compose.ui.Modifier
import androidx.compose.ui.draw.clip
import androidx.compose.ui.graphics.Color
import androidx.compose.ui.graphics.ColorFilter
import androidx.compose.ui.layout.ContentScale
import androidx.compose.ui.res.painterResource
import androidx.compose.ui.unit.dp
import coil.compose.AsyncImage
import com.miam.core.localisation.Localisation
import com.miam.kmm_miam_sdk.android.ressource.Image
import com.miam.kmm_miam_sdk.android.ressource.Image.swap
import com.miam.kmm_miam_sdk.android.theme.Colors
import com.miam.kmm_miam_sdk.android.theme.Colors.primary
import com.miam.kmm_miam_sdk.android.theme.Typography
import com.miam.kmm_miam_sdk.android.ui.components.common.Clickable
import com.miam.sdk.components.price.ProductPrice


class FoundProductImp: FoundProduct {

    private fun newValueBounded(newValue: Int): Boolean {
        return (newValue >= 0) && (newValue <= 100)
    }

    private fun changedValue(localCount: Int?, delta: Int): Int? {

        if (localCount == null) return 0

        if (!newValueBounded(localCount + delta)) return null
        return localCount + delta
    }

    @Composable
    override fun Content(params: FoundProductParameters) {

        Column {
            Spacer(modifier = Modifier.padding(vertical = 8.dp))
            Row(Modifier.padding(horizontal = 16.dp)) {
                Spacer(modifier = Modifier.padding(vertical = 8.dp))
                ProductLine(params)
            }
        }
    }

    @Composable
    fun ProductLine(params: FoundProductParameters) {
        Spacer(modifier = Modifier.padding(vertical = 8.dp))
        Column {
            Row(
                modifier = Modifier.fillMaxWidth(),
                verticalAlignment = Alignment.Top,
                horizontalArrangement = Arrangement.SpaceBetween
            ) {
                AsyncImage(
                    model = params.productPicture,
                    contentDescription = "Product Image",
                    contentScale = ContentScale.Crop,
                    modifier = Modifier
                        .height(80.dp)
                        .width(80.dp)
                        .clip(RoundedCornerShape(8.dp))
                )
                Spacer(modifier = Modifier.padding(horizontal = 8.dp))
                Column {
                    Text(
                        text = params.ingredientName,
                        style = Typography.bodyBold
                    )
                    Spacer(modifier = Modifier.padding(vertical = 2.dp))
                    Text(
                        text = "${params.productName} \n ${params.productCapacityUnit}",
                        color = Colors.grey,
                        style = Typography.bodySmallBold,
                        modifier = Modifier.widthIn(200.dp, 200.dp)
                    )
                    Spacer(modifier = Modifier.padding(vertical = 2.dp))
                    if (params.shareInRecipeCount > 1) {
                        MultipleRecipeTag(params.shareInRecipeCount)
                    }
                }
                IconButton(
                    modifier = Modifier.size(30.dp),
                    onClick = {
                        if (params.isReloading) return@IconButton
                        params.delete()
                    }
                ) {
                    Image(
                        painter = painterResource(Image.delete),
                        contentDescription = "delete"
                    )
                }
            }

            Column(Modifier.fillMaxWidth()) {
                Spacer(modifier = Modifier.padding(vertical = 4.dp))
                EntryPriceAndActionRow(
                    ingredientId = params.ingredientId,
                    price = params.price * params.quantity,
                    currentEntryCount = params.quantity,
                    isReloading = params.isReloading,
                    onCounterChanged = { params.onQuantityChanged(it) }
                ) { params.replace() }
                Spacer(modifier = Modifier.padding(vertical = 4.dp))
                Divider(Modifier.fillMaxWidth(), color = Color.LightGray)
            }
        }
    }

    @Composable
    fun EntryPriceAndActionRow(
        ingredientId: String,
        price: Double,
        currentEntryCount: Int,
        isReloading: Boolean,
        onCounterChanged: (counterValue: Int) -> Unit,
        replace: () -> Unit
    ) {
        Row(
            modifier = Modifier.fillMaxWidth(),
            horizontalArrangement = Arrangement.SpaceBetween,
            verticalAlignment = Alignment.CenterVertically
        ) {
            Clickable(
                onClick = {
                    if (isReloading) return@Clickable
                    replace()
                },
                children = {
                    Row(Modifier) {
                        Image(
                            painter = painterResource(swap),
                            contentDescription = "swap",
                            modifier = Modifier.size(18.dp)
                        )
                        Spacer(modifier = Modifier.padding(horizontal = 2.dp))
                        Text(
                            text = Localisation.Basket.swapProduct.localised,
                            color = primary,
                            style = Typography.body,
                            modifier = Modifier
                        )
                    }
                }
            )

            Spacer(
                modifier = Modifier.weight(1f)
            )
            Box(modifier = Modifier.padding(end = 16.dp)) {
                ProductPrice(price = price)
            }
            Counter(currentEntryCount, isReloading, onCounterChanged, ingredientId)
        }
    }

    @Composable
    private fun Counter(
        initialCount: Int,
        isLoading: Boolean,
        onCounterChanged: (Int) -> Unit,
        key: Any? = null
    ) {
        var mealCountState by remember("$key$initialCount") { mutableStateOf(initialCount) }

        fun increase() {
            changedValue(mealCountState, 1)?.let { newCount ->
                mealCountState = newCount
                onCounterChanged(newCount)
            }
        }

        fun decrease() {
            changedValue(mealCountState, -1)?.let { newCount ->
                mealCountState = newCount
                onCounterChanged(newCount)
            }
        }

        Row(
            verticalAlignment = Alignment.CenterVertically,
            modifier = Modifier.border(BorderStroke(1.dp, primary), shape = RoundedCornerShape(50.dp))
        ) {
            CounterButton(Image.less, isLoading) { decrease() }
            CounterText(mealCountState, isLoading)
            CounterButton(Image.plus, isLoading) { increase() }
        }
    }

    @Composable
    private fun CounterButton(image: Int, isDisable: Boolean, action: () -> Unit) {
        IconButton(
            onClick = { action() },
            enabled = !isDisable,
            modifier = Modifier.background(if (isDisable) Colors.lightgrey else Colors.white)
        ) {
            Image(
                painter = painterResource(image),
                contentDescription = "less icon",
                colorFilter = ColorFilter.tint(primary),
                modifier = Modifier.size(14.dp)
            )
        }
    }

    @Composable
    private fun CounterText(localCount: Int, isLoading: Boolean) {
        Row(
            modifier = Modifier.width(60.dp),
            verticalAlignment = Alignment.CenterVertically,
            horizontalArrangement = Arrangement.Center,
        ) {
            if (isLoading) {
                CircularProgressIndicator(Modifier.size(20.dp), color = primary)
            } else {
                Text(
                    text = localCount.toString(),
                    color = Colors.black,
                    style = Typography.bodyBold
                )
            }
        }
    }

    @Composable
    fun MultipleRecipeTag(sharingCount: Int) {
        if (sharingCount > 1) {
            Surface(
                shape = RoundedCornerShape(8.dp),
                modifier = Modifier.height(20.dp)
            ) {
                Row(
                    verticalAlignment = Alignment.CenterVertically,
                    modifier = Modifier.background(color = Colors.lightgrey)
                ) {
                    Text(
                        modifier = Modifier.padding(horizontal = 8.dp),
                        text = Localisation.Basket.forRecipe(sharingCount).localised,
                        color = Colors.grey,
                        style = Typography.overLine
                    )
                }
            }
        } else {
            Spacer(modifier = Modifier.height(20.dp))
        }
    }
}