package com.miam.sdk.components.price

import androidx.compose.foundation.layout.Box
import androidx.compose.runtime.Composable
import androidx.compose.runtime.DisposableEffect
import androidx.compose.runtime.LaunchedEffect
import androidx.compose.runtime.collectAsState
import androidx.compose.runtime.getValue
import androidx.compose.runtime.mutableStateOf
import androidx.compose.runtime.remember
import androidx.compose.runtime.setValue
import com.miam.core.localisation.Localisation
import com.miam.core.viewModels.pricing.RecipePricingViewModel
import com.miam.sdk.components.baseComponent.ManagementResourceState
import com.miam.sdk.components.price.productPrice.ProductPriceParameters
import com.miam.sdk.components.price.recipePrice.RecipePriceParameters
import com.miam.sdk.di.TemplateDI
import java.text.NumberFormat
import java.util.Currency

fun Double.formatPrice(): String {
    val numberFormat = NumberFormat.getCurrencyInstance()
    numberFormat.currency = Currency.getInstance(Localisation.Price.currency.localised)
    return numberFormat.format(this)
}

@Composable
fun ProductPrice(price: Double) {
    TemplateDI.price.productPrice.view.Content(params = ProductPriceParameters(price))
}

@Composable
fun RecipePrice(
    recipeId: String,
    guestNumber: Int,
) {
    val vmPrice = RecipePricingViewModel()
    vmPrice.setRecipe(recipeId, guestNumber)
    PriceStateManager(vmPrice)
    LaunchedEffect(Unit) { vmPrice.listenBasketChanges() }
    DisposableEffect(Unit) { onDispose { vmPrice.dispose() } }
}

@Composable
fun PriceStateManager(
    vmPrice: RecipePricingViewModel,
) {
    val state by vmPrice.uiState.collectAsState()
    var isLoaded by remember { mutableStateOf(false) }
    Box {
        ManagementResourceState(resourceState = state.price,
            successView = { price ->
                requireNotNull(price)
                LaunchedEffect(price) { isLoaded = true }
                TemplateDI.price.recipePrice.view.Content(params = RecipePriceParameters(price.pricePerServe))
            },
            emptyView = { TemplateDI.price.emptyPrice.view.Content() },
            loadingView = { TemplateDI.price.loadingPrice.view.Content() }
        )
    }
}


