package com.miam.sdk.components.mealPlanner.basketPreview.success.recipeRow

import androidx.compose.animation.core.animateFloatAsState
import androidx.compose.foundation.BorderStroke
import androidx.compose.foundation.Image
import androidx.compose.foundation.background
import androidx.compose.foundation.border
import androidx.compose.foundation.clickable
import androidx.compose.foundation.layout.Arrangement
import androidx.compose.foundation.layout.Column
import androidx.compose.foundation.layout.Row
import androidx.compose.foundation.layout.Spacer
import androidx.compose.foundation.layout.fillMaxWidth
import androidx.compose.foundation.layout.height
import androidx.compose.foundation.layout.padding
import androidx.compose.foundation.layout.size
import androidx.compose.foundation.layout.width
import androidx.compose.foundation.shape.RoundedCornerShape
import androidx.compose.material.CircularProgressIndicator
import androidx.compose.material.Divider
import androidx.compose.material.ExperimentalMaterialApi
import androidx.compose.material.IconButton
import androidx.compose.material.Surface
import androidx.compose.material.Text
import androidx.compose.runtime.Composable
import androidx.compose.runtime.collectAsState
import androidx.compose.runtime.getValue
import androidx.compose.runtime.mutableStateOf
import androidx.compose.runtime.remember
import androidx.compose.runtime.setValue
import androidx.compose.ui.Alignment
import androidx.compose.ui.Modifier
import androidx.compose.ui.draw.clip
import androidx.compose.ui.draw.rotate
import androidx.compose.ui.graphics.ColorFilter
import androidx.compose.ui.layout.ContentScale
import androidx.compose.ui.res.painterResource
import androidx.compose.ui.text.TextStyle
import androidx.compose.ui.text.font.FontWeight
import androidx.compose.ui.text.style.TextOverflow
import androidx.compose.ui.unit.dp
import androidx.compose.ui.unit.sp
import coil.compose.AsyncImage
import com.miam.core.localisation.Localisation
import com.miam.kmm_miam_sdk.android.ressource.Image
import com.miam.kmm_miam_sdk.android.theme.Colors
import com.miam.kmm_miam_sdk.android.theme.Typography
import com.miam.sdk.components.price.formatPrice

class MealPlannerBasketPreviewSuccessRecipeRowImp: MealPlannerBasketPreviewSuccessRecipeRow {


    private fun newValueBounded(newValue: Int): Boolean {
        return (newValue >= 1) && (newValue <= 100)
    }

    private fun changedValue(localCount: Int?, delta: Int): Int? {

        if (localCount == null) return 1

        if (!newValueBounded(localCount + delta)) return null

        return localCount + delta
    }

    @Composable
    fun CollapseButton(initaleState: Boolean, action: () -> Unit) {

        var expandedState by remember { mutableStateOf(initaleState) }
        val rotationState by animateFloatAsState(targetValue = if (expandedState) 90f else 0f, label = "rotation state")

        IconButton(
            modifier = Modifier
                .size(30.dp)
                .rotate(rotationState),
            onClick = {
                expandedState = !expandedState
                action()
            }
        ) {
            Image(
                painter = painterResource(Image.toggleCaret),
                contentDescription = null,
                modifier = Modifier.size(24.dp),
                colorFilter = ColorFilter.tint(Colors.primary),
            )
        }
    }

    @Composable
    fun CounterButton(image: Int, isDisable: Boolean, action: () -> Unit) {

        IconButton(
            onClick = { action() },
            enabled = !isDisable,
            modifier = Modifier
                .clip(RoundedCornerShape(8.dp))
                .background(if (isDisable) Colors.lightgrey else Colors.white)
        ) {
            Image(
                painter = painterResource(image),
                contentDescription = "less icon",
                colorFilter = ColorFilter.tint(Colors.primary),
                modifier = Modifier.size(14.dp)
            )
        }
    }

    @Composable
    fun CounterText(localCount: Int?, isLoading: Boolean) {

        Row(
            modifier = Modifier,
            verticalAlignment = Alignment.CenterVertically,
            horizontalArrangement = Arrangement.Center,
        ) {
            if (isLoading) {
                CircularProgressIndicator(color = Colors.white)
            } else {
                Text(
                    text = localCount.toString() + " " + Localisation.Counter.persons.localised,
                    color = Colors.black,
                    style = Typography.bodyBold
                )
            }
        }
    }


    @Composable
    fun Counter(initialCount: Int, isLoading: Boolean, onCounterChanged: (Int) -> Unit) {

        var localCount by remember { mutableStateOf(initialCount) }

        fun increase() {
            changedValue(localCount, 1)?.let { newCount ->
                localCount = newCount
                onCounterChanged(newCount)
            }
        }

        fun decrease() {
            changedValue(localCount, -1)?.let { newCount ->
                localCount = newCount
                onCounterChanged(newCount)
            }
        }

        Row(
            verticalAlignment = Alignment.CenterVertically,
            modifier = Modifier.border(BorderStroke(1.dp, Colors.primary), shape = RoundedCornerShape(8.dp))
        ) {
            CounterButton(Image.less, isLoading) { decrease() }
            CounterText(localCount, isLoading)
            CounterButton(Image.plus, isLoading) { increase() }
        }
    }

    @Composable
    override fun Content(params: MealPlannerBasketPreviewSuccessRecipeRowParameters) {

        val isLocked = params.isLock.collectAsState()

        Column {
            Row(
                verticalAlignment = Alignment.Top,
                modifier = Modifier
                    .fillMaxWidth()
                    .padding(12.dp)
            ) {
                AsyncImage(
                    model = params.picture,
                    contentDescription = null,
                    contentScale = ContentScale.Crop,
                    modifier = Modifier
                        .height(120.dp)
                        .width(120.dp)
                        .clip(RoundedCornerShape(8.dp))
                        .clickable {
                            if (params.isDeleting || isLocked.value) return@clickable
                            params.openRecipeDetail()
                        }
                )
                Spacer(modifier = Modifier.padding(horizontal = 16.dp))
                Column(
                    verticalArrangement = Arrangement.Top
                ) {
                    Row(
                        Modifier.fillMaxWidth(),
                        verticalAlignment = Alignment.Top,
                        horizontalArrangement = Arrangement.SpaceBetween
                    ) {
                        Text(
                            text = params.name,
                            maxLines = 2,
                            color = Colors.black,
                            overflow = TextOverflow.Ellipsis,
                            style = Typography.bodyBold,
                            modifier = Modifier
                                .weight(1f)
                                .clickable {
                                    if (params.isDeleting) return@clickable
                                    params.openRecipeDetail()
                                }
                        )
                        Column() {
                            Spacer(modifier = Modifier.height(8.dp))
                            if (params.isDeleting) {
                                CircularProgressIndicator(Modifier.size(20.dp), Colors.primary, 1.dp)
                            } else {
                                IconButton(
                                    modifier = Modifier.size(20.dp),
                                    onClick = {
                                        if (isLocked.value) return@IconButton
                                        params.delete()
                                    }
                                ) {
                                    Image(
                                        painter = painterResource(Image.delete),
                                        contentDescription = "delete",
                                        colorFilter = ColorFilter.tint(Colors.primary),
                                    )
                                }
                            }
                        }
                    }
                    Spacer(modifier = Modifier.padding(vertical = 4.dp))
                    if (!params.isChildrenLoading) {
                        Text(
                            text = Localisation.Basket.articles(params.productCount).localised,
                            color = Colors.grey,
                            style = Typography.bodySmall
                        )
                    }
                    SeeDetailsButton(
                        text = Localisation.Recipe.showDetails.localised,
                        isLocked = isLocked.value,
                        params.openRecipeDetail
                    )
                    Spacer(modifier = Modifier.padding(vertical = 4.dp))
                }
            }
            Row(
                Modifier
                    .fillMaxWidth()
                    .padding(16.dp),
                horizontalArrangement = Arrangement.SpaceBetween,
                verticalAlignment = Alignment.CenterVertically
            ) {
                if (params.isChildrenLoading || params.isPriceRefreshing || isLocked.value) {
                    CircularProgressIndicator(Modifier.size(20.dp), Colors.primary, 1.dp)
                } else {
                    PriceInfo(
                        params.price,
                        params.numberOfGuest
                    )
                }
                Row(verticalAlignment = Alignment.CenterVertically) {
                    Counter(
                        initialCount = params.numberOfGuest,
                        isLoading = params.isDeleting
                    ) {
                        params.onGuestChange(it)
                    }
                    Spacer(modifier = Modifier.padding(horizontal = 8.dp))
                    CollapseButton(params.isExpanded) {
                        params.toggleCollapse()
                    }
                }
            }
            Divider()
        }
    }
}

@OptIn(ExperimentalMaterialApi::class)
@Composable
fun SeeDetailsButton(text: String, isLocked: Boolean, action: () -> Unit = {}) {
    Surface(
        shape = RoundedCornerShape(10.dp),
        border = BorderStroke(1.dp, Colors.primary),
        color = Colors.white,
        onClick = {
            if (isLocked) return@Surface
            action()
        }) {
        Row(
            Modifier.padding(horizontal = 8.dp, vertical = 4.dp),
            verticalAlignment = Alignment.CenterVertically,
            horizontalArrangement = Arrangement.Center
        ) {
            Text(text = text, style = TextStyle(fontSize = 16.sp, color = Colors.primary, fontWeight = FontWeight.Black))
        }
    }
}

@Composable
fun PriceInfo(price: Double, guestCount: Int) {
    Column(modifier = Modifier.padding(bottom = 4.dp)) {
        Column(
            horizontalAlignment = Alignment.CenterHorizontally
        ) {
            Row() {
                Text(
                    price.formatPrice(),
                    color = Colors.primary,
                    style = Typography.subtitleBold
                )
            }
        }
        Text(
            text = "${(price / guestCount).formatPrice()} ${Localisation.Price.perGuest.localised}",
            color = Colors.grey,
            style = Typography.overLine
        )
    }
}