package com.miam.sdk.components.itemSelector

import android.content.Context
import android.util.AttributeSet
import androidx.compose.foundation.layout.Column
import androidx.compose.foundation.layout.padding
import androidx.compose.foundation.rememberScrollState
import androidx.compose.foundation.verticalScroll
import androidx.compose.material.Scaffold
import androidx.compose.material.Surface
import androidx.compose.runtime.Composable
import androidx.compose.runtime.DisposableEffect
import androidx.compose.runtime.LaunchedEffect
import androidx.compose.runtime.collectAsState
import androidx.compose.runtime.getValue
import androidx.compose.ui.Modifier
import androidx.compose.ui.platform.AbstractComposeView
import com.miam.core.localisation.Localisation
import com.miam.core.viewModels.itemSelector.ItemSelectorContract
import com.miam.core.viewModels.itemSelector.ItemSelectorEmptyStates
import com.miam.core.viewModels.itemSelector.ItemSelectorViewModel
import com.miam.core.viewModels.quantityFormatter.QuantityFormatter
import com.miam.sdk.components.baseComponent.ManagementResourceState
import com.miam.sdk.components.baseComponent.emptyPage.EmptyPageParameters
import com.miam.sdk.components.itemSelector.header.ItemSelectorHeaderParameters
import com.miam.sdk.components.itemSelector.search.ItemSelectorSearchParameters
import com.miam.sdk.components.itemSelector.selectedItem.ItemSelectorSelectedItemParameters
import com.miam.sdk.components.itemSelector.success.ItemSelectorSuccessParameters
import com.miam.sdk.di.TemplateDI

class ItemSelector @JvmOverloads constructor(
    context: Context,
    attrs: AttributeSet? = null,
    defStyleAttr: Int = 0
): AbstractComposeView(context, attrs, defStyleAttr) {

    var ingredientId = ""
        set(value) {
            field = value
        }
    var back: () -> Unit = { error("You need to set a back function on ItemSelector") }
        set(value) {
            field = {
                vmItemSelector.setEvent(ItemSelectorContract.Event.ReturnToDetail)
                value()
            }
        }

    private val vmItemSelector = ItemSelectorViewModel()

    @Composable
    override fun Content() {
        View(vmItemSelector) {
            back()
        }
    }

    companion object {
        @Composable
        fun View(vmItemSelector: ItemSelectorViewModel, ingredientId: String? = null, back: () -> Unit) {

            DisposableEffect(Unit) { onDispose { vmItemSelector.dispose() } }
            LaunchedEffect(Unit) {
                if (ingredientId != null) {
                    vmItemSelector.registerListeners()
                    vmItemSelector.setEvent(ItemSelectorContract.Event.InitWithIngredientId(ingredientId))
                }
            }

            val state by vmItemSelector.uiState.collectAsState()

            Scaffold(
                topBar = {
                    TemplateDI.itemSelector.header.view.Content(
                        params = ItemSelectorHeaderParameters { back() }
                    )
                }
            ) { scaffoldPadding ->
                Column(Modifier.padding(scaffoldPadding)) {
                    TemplateDI.itemSelector.search.view.Content(params = ItemSelectorSearchParameters(
                        state.ingredientName,
                        QuantityFormatter.realQuantities(state.ingredientQuantity.toString(), state.guestCount, state.defaultRecipeGuest).toInt(),
                        state.ingredientUnit
                    ) {
                        vmItemSelector.setEvent(ItemSelectorContract.Event.Search(it))
                    })
                    Column(modifier = Modifier.verticalScroll(rememberScrollState())) {
                        state.selectedItem?.let {
                            TemplateDI.itemSelector.selectedItem.view.Content(params = ItemSelectorSelectedItemParameters(it))
                        }
                        ManagementResourceState(
                            resourceState = state.items,
                            successView = { items ->
                                requireNotNull(items)
                                Surface {
                                    TemplateDI.itemSelector.success.view.Content(params = ItemSelectorSuccessParameters(
                                        items = items,
                                        previous = {
                                            vmItemSelector.setEvent(ItemSelectorContract.Event.ReturnToDetail)
                                            back()
                                        },
                                        select = { item ->
                                            vmItemSelector.setEvent(ItemSelectorContract.Event.SelectItem(item))
                                            back()
                                        }
                                    ))
                                }
                            },
                            loadingView = {
                                TemplateDI.itemSelector.loading.view.Content()
                            },
                            emptyView = {
                                val title = if (vmItemSelector.emptyState == ItemSelectorEmptyStates.NO_RESULTS_AFTER_SEARCH) {
                                    Localisation.ItemSelector.notFoundTitle.localised
                                } else {
                                    Localisation.ItemSelector.noSubstitution.localised
                                }
                                val subtitle = if (vmItemSelector.emptyState == ItemSelectorEmptyStates.NO_RESULTS_AFTER_SEARCH) {
                                    Localisation.ItemSelector.notFoundSubtitle.localised
                                } else ""
                                val params = EmptyPageParameters(
                                    title = title,
                                    subtitle = subtitle,
                                )
                                TemplateDI.itemSelector.empty?.view?.Content(params = params)
                                    ?: TemplateDI.defaultViews.empty.view.Content(params = params)
                            }
                        )
                    }
                }
            }
        }
    }
}