package com.miam.sdk.components.recipeJourney

import android.content.Context
import android.util.AttributeSet
import androidx.compose.runtime.Composable
import androidx.compose.runtime.collectAsState
import androidx.compose.runtime.mutableStateOf
import androidx.compose.runtime.remember
import androidx.compose.ui.ExperimentalComposeUiApi
import androidx.compose.ui.platform.AbstractComposeView
import androidx.compose.ui.platform.LocalContext
import androidx.compose.ui.window.Dialog
import androidx.compose.ui.window.DialogProperties
import androidx.navigation.compose.NavHost
import androidx.navigation.compose.composable
import androidx.navigation.compose.rememberNavController
import com.miam.core.model.Recipe
import com.miam.core.model.SuggestionsCriteria
import com.miam.core.viewModels.dynamicRecipeDetail.DynamicRecipeDetailViewModel
import com.miam.core.viewModels.itemSelector.ItemSelectorContract
import com.miam.core.viewModels.itemSelector.ItemSelectorViewModel
import com.miam.core.viewModels.recipe.RecipeContract
import com.miam.core.viewModels.recipe.RecipeViewModel
import com.miam.sdk.components.itemSelector.ItemSelector
import com.miam.sdk.components.recipeCard.RecipeCard
import com.miam.sdk.components.recipeDetail.RecipeDetail
import com.miam.sdk.components.sponsorDetail.SponsorDetail
import kotlinx.coroutines.flow.MutableStateFlow

class RecipeJourney @JvmOverloads constructor(
    context: Context,
    attrs: AttributeSet? = null,
    defStyleAttr: Int = 0
): AbstractComposeView(context, attrs, defStyleAttr) {

    private data class RecipeJourneyState(
        val recipe: Recipe? = null,
        val criteria: SuggestionsCriteria? = null,
        val journeyIsInShelve: Boolean = true,
        val recipeId: String? = null
    )

    private val state = MutableStateFlow(RecipeJourneyState())
    private var baseComposeView: (@Composable (recipeId: String, goToDetail: (recipeId: String) -> Unit) -> Unit)? = null
    private var cookOnlyMode: Boolean = false

    fun bind(
        recipeId: String? = null,
        criteria: SuggestionsCriteria? = null,
        recipe: Recipe? = null,
        journeyIsInShelve: Boolean = true,
        cookOnlyMode: Boolean = false,
        //baseComposeView: (@Composable (recipeId: String, goToDetail: (recipeId: String) -> Unit) -> Unit)? = null
    ) {
        state.value =
            RecipeJourneyState(journeyIsInShelve = journeyIsInShelve, recipeId = recipeId ?: recipe?.id, criteria = criteria, recipe = recipe)
        //this.baseComposeView = baseComposeView
        this.cookOnlyMode = cookOnlyMode
    }

    @Composable
    override fun Content() {

        val recipeJourneyState = state.collectAsState()
        View(
            recipeId = recipeJourneyState.value.recipeId,
            criteria = recipeJourneyState.value.criteria,
            recipe = recipeJourneyState.value.recipe,
            baseComposeView = baseComposeView,
            cookOnlyMode = cookOnlyMode
        )
    }

    companion object {

        @OptIn(ExperimentalComposeUiApi::class)
        @Composable
        fun View(
            recipeId: String? = null,
            criteria: SuggestionsCriteria? = null,
            recipe: Recipe? = null,
            cookOnlyMode: Boolean = false,
            isInShelve: Boolean = false,
            baseComposeView: (@Composable (recipeId: String, goToDetail: (recipeId: String) -> Unit) -> Unit)? = null
        ) {
            val navController = rememberNavController()
            val dialogIsOpen = remember { mutableStateOf(false) }
            fun goToItemSelector(ingredientId: String) = navController.navigate("ITEM_SELECTOR/${ingredientId}") { launchSingleTop = true }
            fun goToDetail() {
                dialogIsOpen.value = true
            }

            fun goToSponsor(sponsorId: String) = navController.navigate("SPONSOR/${sponsorId}") { launchSingleTop = true }
            fun back() {
                if (navController.currentBackStackEntry?.destination?.route == "DETAIL") {
                    dialogIsOpen.value = false
                    return
                }
                navController.popBackStack()
            }

            val dynamicRecipeDetailViewModel = remember { DynamicRecipeDetailViewModel(::back, ::goToItemSelector) }
            val recipeViewModel = remember {
                RecipeViewModel().apply {
                    val localRecipeId = recipeId ?: recipe?.id
                    if (localRecipeId != null) {
                        this.setEvent(RecipeContract.Event.FetchRecipeFromId(localRecipeId))
                    } else {
                        criteria?.let {
                            this.setEvent(RecipeContract.Event.FetchRecipeFromCriteria(criteria))
                        } ?: error("recipe need an id or a criteria to start")
                    }
                    this.goToDetail = { _ -> goToDetail() }
                }
            }
            val itemSelectorViewModel = remember { ItemSelectorViewModel() }

            baseComposeView?.let {
                val localRecipeId = recipeId ?: recipe?.id

                localRecipeId?.let { recipeId ->
                    it(recipeId) { _ -> goToDetail() }
                }

            } ?: run {
                RecipeCard.View(recipeViewModel, isInShelve = isInShelve)
            }

            if (dialogIsOpen.value) {
                Dialog(
                    onDismissRequest = ::back,
                    properties = DialogProperties(usePlatformDefaultWidth = false)
                ) {
                    recipeViewModel.currentState.recipe?.id?.let { recipeId ->
                        NavHost(navController = navController, startDestination = "DETAIL") {
                            composable("DETAIL") {
                                println("Perf miam RecipeJourney DETAIL printing")
                                RecipeDetail.View(
                                    viewModel = dynamicRecipeDetailViewModel,
                                    recipeId = recipeId,
                                    goToSponsor = ::goToSponsor,
                                    cookOnlyMode = cookOnlyMode
                                )
                            }
                            composable("ITEM_SELECTOR/{ingredientId}") { backStackEntry ->
                                backStackEntry.arguments?.let { arguments ->
                                    ItemSelector.View(itemSelectorViewModel, arguments.getString("ingredientId")) {
                                        itemSelectorViewModel.setEvent(ItemSelectorContract.Event.ReturnToDetail)
                                        back()
                                    }
                                }
                            }
                            composable("SPONSOR/{sponsorId}") { backStackEntry ->
                                backStackEntry.arguments?.getString("sponsorId")?.let { sponsorId ->
                                    SponsorDetail(LocalContext.current).apply { bind(sponsorId, ::back) }.Content()
                                }
                            }
                        }
                    }
                }
            }
        }
    }
}