package com.miam.sdk.components.mealPlanner.meals


import android.content.Context
import android.util.AttributeSet
import androidx.compose.animation.AnimatedVisibility
import androidx.compose.animation.core.tween
import androidx.compose.animation.fadeIn
import androidx.compose.animation.fadeOut
import androidx.compose.animation.slideInHorizontally
import androidx.compose.foundation.layout.Box
import androidx.compose.foundation.layout.Column
import androidx.compose.foundation.layout.padding
import androidx.compose.foundation.rememberScrollState
import androidx.compose.foundation.verticalScroll
import androidx.compose.material.ExperimentalMaterialApi
import androidx.compose.material.Scaffold
import androidx.compose.material.pullrefresh.PullRefreshIndicator
import androidx.compose.material.pullrefresh.pullRefresh
import androidx.compose.material.pullrefresh.rememberPullRefreshState
import androidx.compose.runtime.Composable
import androidx.compose.runtime.DisposableEffect
import androidx.compose.runtime.collectAsState
import androidx.compose.runtime.getValue
import androidx.compose.runtime.mutableStateOf
import androidx.compose.runtime.remember
import androidx.compose.runtime.rememberCoroutineScope
import androidx.compose.runtime.setValue
import androidx.compose.ui.Alignment
import androidx.compose.ui.Modifier
import androidx.compose.ui.graphics.Color
import androidx.compose.ui.platform.AbstractComposeView
import androidx.compose.ui.zIndex
import com.miam.core.base.state.ComponentUiState
import com.miam.core.data.repository.BudgetRepository
import com.miam.core.handler.LogHandler
import com.miam.core.model.MealPlannerRecipe
import com.miam.core.viewModels.mealPlanner.form.MealPlannerFormViewModel
import com.miam.core.viewModels.mealPlanner.meals.MealPlannerMealsViewModel
import com.miam.sdk.components.baseComponent.ManagementResourceState
import com.miam.sdk.components.mealPlanner.form.success.MealPlannerFormSuccessParameters
import com.miam.sdk.components.mealPlanner.meals.empty.MealsEmptyParameters
import com.miam.sdk.components.mealPlanner.meals.footer.MealsFooterParameters
import com.miam.sdk.components.mealPlanner.meals.loading.MealsLoadingParameters
import com.miam.sdk.components.mealPlanner.recipe.MealPlannerRecipeCard
import com.miam.sdk.components.mealPlanner.recipe.success.placeholder.MealPlannerRecipePlaceholderParameters
import com.miam.sdk.di.TemplateDI
import kotlinx.coroutines.CoroutineExceptionHandler
import kotlinx.coroutines.launch

class MealPlanner @JvmOverloads constructor(
    context: Context,
    attrs: AttributeSet? = null,
    defStyleAttr: Int = 0
): AbstractComposeView(context, attrs, defStyleAttr) {
    private val mealsViewModel = MealPlannerMealsViewModel()
    private val formViewModel = MealPlannerFormViewModel()
    var goToRecipeSelector = { _: Int -> }
    var onComfirm = {}

    private val coroutineHandler = CoroutineExceptionHandler { _, exception ->
        LogHandler.error("Miam error in MealPlanner $exception")
    }

    private suspend fun loadRecipe(budget: Int, numberOfMeals: Int, numberOfGuests: Int) {
        try {
            formViewModel.getRecipesForBudgetConstraint(budget, numberOfMeals, numberOfGuests)
            if (formViewModel.currentState.uiState != ComponentUiState.SUCCESS) return
        } catch (e: Throwable) {
            formViewModel.getRecipesMaxCountForBudgetConstraint(budget, numberOfMeals)
        }
    }

    @OptIn(ExperimentalMaterialApi::class)
    @Composable
    override fun Content() {
        DisposableEffect(Unit) { onDispose { mealsViewModel.dispose() } }

        val refreshScope = rememberCoroutineScope()
        val formState by formViewModel.uiState.collectAsState()
        var refreshing by remember { mutableStateOf(false) }


        val pullState = rememberPullRefreshState(refreshing, {
            refreshScope.launch(coroutineHandler) {
                refreshing = true
                mealsViewModel.onPullToRefresh()
                refreshing = false
            }
        })

        Scaffold(backgroundColor = Color.Transparent, bottomBar = { footerSection() }) {
            Column(
                Modifier
                    .padding(it)
                    .pullRefresh(pullState)
                    .verticalScroll(rememberScrollState()), horizontalAlignment = Alignment.CenterHorizontally
            ) {
                Box(Modifier.zIndex(10f)) {
                    TemplateDI.mealPlanner.toolbar.view.Content(
                        params = MealPlannerFormSuccessParameters(
                            budget = formState.budget,
                            numberOfGuests = formState.numberOfGuests,
                            numberOfMeals = formState.numberOfMeals,
                            uiState = formState.uiState,
                            setBudget = { budget -> formViewModel.setBudget(budget) },
                            setNumberOfGuests = { nbGuest -> formViewModel.setNumberOfGuests(nbGuest) },
                            setNumberOfMeals = { nbMeal -> formViewModel.setNumberOfMeals(nbMeal) },
                            maxMealCount = formState.recipeMaxCount,
                            currentRemainingRecipeCountFlow = formViewModel.recipeCountFlow,
                            refreshMaxMealCount = { budget, numberOfGuests ->
                                formViewModel.getRecipesMaxCountForBudgetConstraint(
                                    budget,
                                    numberOfGuests
                                )
                            },
                        ) { budget, numberOfMeals, numberOfGuests ->
                            refreshScope.launch(coroutineHandler) { loadRecipe(budget, numberOfMeals, numberOfGuests) }
                        }
                    )
                }
                PullRefreshIndicator(refreshing, pullState, Modifier.zIndex(9f))
                Meals()
            }
        }
    }

    @Composable
    fun Meals() {
        val state by mealsViewModel.uiState.collectAsState()
        ManagementResourceState(
            resourceState = state.meals,
            successView = { meals ->
                requireNotNull(meals)
                meals.forEachIndexed { index, recipe ->
                    MealPlannerCardSection(
                        recipe = recipe,
                        index = index,
                        context = context,
                        goToChangeRecipe = {
                            mealsViewModel.calculAvailableBudgetOnNavigateToReplaceRecipe(state.totalPrice, recipe?.price)
                            goToRecipeSelector(it)
                        },
                        removeRecipe = { mealsViewModel.removeRecipe(index) }
                    )
                }
            },
            loadingView = { TemplateDI.mealPlanner.loading.view.Content(MealsLoadingParameters(state.numberOfMeal)) },
            emptyView = {
                TemplateDI.mealPlanner.empty.view.Content(
                    MealsEmptyParameters(BudgetRepository.budget, BudgetRepository.guestCount, BudgetRepository.mealCount)
                )
            },
            errorView = {}
        )
    }

    @Composable
    fun MealPlannerCardSection(
        recipe: MealPlannerRecipe?,
        index: Int,
        context: Context,
        goToChangeRecipe: (Int) -> Unit,
        removeRecipe: () -> Unit
    ) {

        val localRecipe = remember(recipe?.recipeId) { recipe?.recipeId }
        val recipeCard = remember(localRecipe) {
            localRecipe?.let {
                MealPlannerRecipeCard(context).apply {
                    idRecipe = it
                    price = recipe?.price ?: 0.0
                    changeAction = { goToChangeRecipe(index) }
                    deleteAction = removeRecipe
                }
            }
        }
        AnimatedVisibility(
            visible = recipe != null,
            enter = fadeIn(animationSpec = tween(delayMillis = (100 * index))),
            exit = fadeOut(animationSpec = tween(durationMillis = 0))
        ) { recipeCard?.Content() }


        AnimatedVisibility(
            visible = recipe == null,
            enter = slideInHorizontally(),
            exit = fadeOut(animationSpec = tween(durationMillis = 0))
        ) {
            TemplateDI.mealPlanner.recipePlaceholder.view.Content(
                mealPlannerRecipePlaceholderParameters = MealPlannerRecipePlaceholderParameters { goToChangeRecipe(index) }
            )
        }
    }


    @Composable
    fun footerSection() {
        val state by mealsViewModel.uiState.collectAsState()
        TemplateDI.mealPlanner.footer.view.Content(budgetPlannerFooterParameters =
        MealsFooterParameters(
            budget = BudgetRepository.budget,
            recipesPrice = state.totalPrice,
            recipeCount = BudgetRepository.mealCount,
            addAllToBasket = {
                onComfirm()
                mealsViewModel.addRecipesToGroceriesList()
            }
        ))
    }
}

