package com.miam.sdk.components.catalog.success

import androidx.compose.foundation.layout.Box
import androidx.compose.foundation.layout.Column
import androidx.compose.foundation.layout.padding
import androidx.compose.runtime.Composable
import androidx.compose.runtime.getValue
import androidx.compose.runtime.mutableStateOf
import androidx.compose.runtime.remember
import androidx.compose.runtime.setValue
import androidx.compose.ui.Modifier
import androidx.compose.ui.platform.LocalContext
import androidx.compose.ui.unit.dp
import com.miam.core.localisation.Localisation
import com.miam.core.model.Package
import com.miam.core.services.RouteServiceInstance
import com.miam.core.viewModels.catalog.CatalogContent
import com.miam.core.viewModels.catalog.DialogContent
import com.miam.kmm_miam_sdk.android.ui.components.common.RoutableDialog
import com.miam.sdk.components.baseComponent.filter.Filter
import com.miam.sdk.components.baseComponent.recipesPage.RecipesPage
import com.miam.sdk.components.catalog.success.categoriesPage.CatalogCategoriesPage
import com.miam.sdk.components.catalog.success.floating.CatalogFloatingParameters
import com.miam.sdk.components.catalog.success.search.CatalogSuccessSearchParameters
import com.miam.sdk.components.catalog.success.toolbar.CatalogSuccessToolbarParameters
import com.miam.sdk.components.preferences.Preferences
import com.miam.sdk.di.TemplateDI

@Composable
fun CatalogSuccess(
    categories: List<Package>,
    content: CatalogContent,
    isDialogOpen: Boolean,
    dialogContent: DialogContent,
    currentSearchString: String,
    openFilter: () -> Unit,
    openSearch: () -> Unit,
    openPreferences: () -> Unit,
    getActiveFilterCount: () -> Int,
    onGoToFavorite: () -> Unit,
    onGoToCategoryPage: (categoryId: String, title: String, subtitle: String?) -> Unit,
    search: (CatalogContent) -> Unit,
    onGoToMealPlanner: (() -> Unit)? = null,
    onGoToBasket: (() -> Unit)? = null
) {
    val routeService = RouteServiceInstance.instance
    var currentPackageTitle by remember { mutableStateOf<String?>(null) }

    fun back() = routeService.previous()

    FilterAndPreferenceModal(
        isDialogOpen,
        dialogContent,
        { search(it) },
        { routeService.onCloseDialog() },
        ::back
    )
    Box {
        Column(modifier = Modifier.padding(bottom = 24.dp)) {
            TemplateDI.catalog.success.toolbar.view.Content(
                params = CatalogSuccessToolbarParameters(
                    content = content,
                    openFilter = { openFilter() },
                    openSearch = { openSearch() },
                    goToFavorite = { onGoToFavorite() },
                    openPreferences = { openPreferences() },
                    getActiveFilterCount = { getActiveFilterCount() },
                    goToBack = ::back
                )
            )

            when (content) {
                CatalogContent.CATEGORIES_LIST -> {
                    CatalogCategoriesPage(
                        categories = categories,
                        goToCategoryPage = { categoryId, title, subtitle ->
                            currentPackageTitle = title
                            onGoToCategoryPage(categoryId, title, subtitle)
                        },
                        goToMealPlanner = onGoToMealPlanner,
                    )
                }

                CatalogContent.FILTER_SEARCH -> {
                    RecipesPage(LocalContext.current).apply {
                        bind(
                            Localisation.Catalog.searchTitle.localised,
                            ::back,
                            Localisation.Catalog.noRecipeFoundFilter.localised,
                            Localisation.Catalog.tryWithOtherFilter.localised
                        )
                    }.Content()
                }

                CatalogContent.WORD_SEARCH -> {
                    val title = "${Localisation.Catalog.searchPrefix.localised} \"${currentSearchString}\""
                    RecipesPage(LocalContext.current).apply {
                        bind(
                            title,
                            ::back,
                            Localisation.Catalog.noRecipeFound.localised + " $currentSearchString",
                            Localisation.Catalog.tryAnotherSearch.localised
                        )
                    }.Content()
                }

                CatalogContent.CATEGORY -> {
                    val title = currentPackageTitle
                    RecipesPage(LocalContext.current).apply {
                        bind(
                            title ?: Localisation.Catalog.searchTitle.localised,
                            ::back,
                            Localisation.Catalog.noRecipeFoundCategory.localised,
                            Localisation.Catalog.tryWithOtherFilterOrPreferences.localised
                        )
                    }.Content()
                }

                CatalogContent.FAVORITE -> RecipesPage(LocalContext.current).apply {
                    bind(
                        Localisation.Catalog.favoriteTitle.localised,
                        { back() },
                        Localisation.Favorites.noFavorites.localised,
                        ""
                    )
                }.Content()
            }
        }
        Box(modifier = Modifier.align(TemplateDI.catalog.success.floating.footerAlignment)) {
            TemplateDI.catalog.success.floating.view.Content(
                params = CatalogFloatingParameters(
                    onClickAction = {
                        onGoToBasket?.let {
                            it()
                        }
                    }
                )
            )
        }
    }
}

@Composable
fun FilterAndPreferenceModal(
    isDialogOpen: Boolean,
    content: DialogContent,
    search: (CatalogContent) -> Unit,
    close: () -> Unit,
    previous: () -> Unit,
) {
    val context = LocalContext.current
    val filter = remember {
        Filter(
            "Catalog",
            { close() },
            { search(CatalogContent.FILTER_SEARCH) }
        )
    }
    val preference = remember {
        Preferences(context).apply {
            bind({ close() }, { close() })
        }
    }

    if (isDialogOpen) {
        RoutableDialog(onDismissRequest = {
            previous()
        }) {
            when (content) {
                DialogContent.FILTER -> filter.Content()
                DialogContent.SEARCH -> TemplateDI.catalog.success.search.view.Content(params = CatalogSuccessSearchParameters(
                    { close() },
                    { search(CatalogContent.WORD_SEARCH) }
                ))

                DialogContent.PREFERENCES -> preference.Content()
            }
        }
    }
}