package com.miam.core.components.catalog

import android.content.Context
import android.util.AttributeSet
import androidx.compose.foundation.layout.Box
import androidx.compose.foundation.layout.Column
import androidx.compose.runtime.Composable
import androidx.compose.runtime.DisposableEffect
import androidx.compose.runtime.collectAsState
import androidx.compose.runtime.getValue
import androidx.compose.ui.platform.AbstractComposeView
import com.miam.core.localisation.Localisation
import com.miam.core.viewModels.catalog.CatalogContract
import com.miam.core.viewModels.catalog.CatalogFilterViewModelFactory
import com.miam.core.viewModels.catalog.CatalogViewModel
import com.miam.sdk.components.baseComponent.ManagementResourceState
import com.miam.sdk.components.baseComponent.emptyPage.EmptyPageParameters
import com.miam.sdk.components.baseComponent.loader.LoaderParameters
import com.miam.sdk.components.catalog.success.CatalogSuccess
import com.miam.sdk.di.TemplateDI

class Catalog @JvmOverloads constructor(
    context: Context,
    attrs: AttributeSet? = null,
    defStyleAttr: Int = 0,
): AbstractComposeView(context, attrs, defStyleAttr) {

    private val filterVM = CatalogFilterViewModelFactory().instantiate("Catalog")
    private val vmCatalog: CatalogViewModel = CatalogViewModel()
    var goToMealPlanner: (() -> Unit)? = null
    var goToBasket: (() -> Unit)? = null

    fun bind(
        categoryId: String? = null,
        title: String? = null,
        goToMealPlanner: (() -> Unit)? = null,
        goToBasket: (() -> Unit)? = null,
    ) {
        if (categoryId != null) {
            vmCatalog.goToCategory(categoryId, title ?: "")
        }
        goToMealPlanner?.let {
            this.goToMealPlanner = it
        }
        goToBasket?.let {
            this.goToBasket = it
        }
    }

    @Composable
    override fun Content() {
        val state by vmCatalog.uiState.collectAsState()

        DisposableEffect(Unit) { onDispose { vmCatalog.dispose() } }

        Box {
            Column {
                ManagementResourceState(
                    resourceState = state.categories,
                    successView = { categories ->
                        requireNotNull(categories)
                        CatalogSuccess(
                            categories = categories,
                            content = state.content,
                            isDialogOpen = state.dialogIsOpen,
                            dialogContent = state.dialogContent,
                            currentSearchString = filterVM.currentState.searchString ?: "",
                            openFilter = { vmCatalog.openFilter() },
                            openSearch = { vmCatalog.openSearch() },
                            openPreferences = { vmCatalog.openPreferences() },
                            getActiveFilterCount = { filterVM.getActiveFilterCount() },
                            onGoToFavorite = {
                                filterVM.setFavorite()
                                vmCatalog.setEvent(CatalogContract.Event.GoToFavorite)
                            },
                            onGoToCategoryPage = { id, title, subtitle ->
                                filterVM.setCat(id)
                                vmCatalog.goToCategory(id, title, subtitle)
                            },
                            search = { vmCatalog.onSimpleSearch(it) },
                            onGoToMealPlanner = goToMealPlanner,
                            onGoToBasket = goToBasket
                        )
                    },
                    loadingView = {
                        TemplateDI.catalog.loader?.view?.Content(params = LoaderParameters)
                            ?: TemplateDI.defaultViews.loading.view.Content(params = LoaderParameters)
                    },
                    emptyView = {
                        val params = EmptyPageParameters(Localisation.Catalog.preferencesNoResult.localised)
                        TemplateDI.catalog.empty?.view?.Content(params = params)
                            ?: TemplateDI.defaultViews.empty.view.Content(params = params)
                    }
                )
            }
        }
    }
}