package com.miam.sdk.components.recipeDetail.success.ingredients

import androidx.compose.foundation.BorderStroke
import androidx.compose.foundation.Image
import androidx.compose.foundation.layout.Arrangement
import androidx.compose.foundation.layout.Box
import androidx.compose.foundation.layout.Column
import androidx.compose.foundation.layout.Row
import androidx.compose.foundation.layout.Spacer
import androidx.compose.foundation.layout.fillMaxWidth
import androidx.compose.foundation.layout.height
import androidx.compose.foundation.layout.padding
import androidx.compose.foundation.layout.width
import androidx.compose.foundation.shape.RoundedCornerShape
import androidx.compose.material.Surface
import androidx.compose.material.Text
import androidx.compose.runtime.Composable
import androidx.compose.ui.Alignment
import androidx.compose.ui.Modifier
import androidx.compose.ui.layout.ContentScale
import androidx.compose.ui.res.painterResource
import androidx.compose.ui.text.TextStyle
import androidx.compose.ui.text.font.FontWeight
import androidx.compose.ui.text.style.TextAlign
import androidx.compose.ui.unit.dp
import androidx.compose.ui.unit.sp
import coil.compose.rememberImagePainter
import com.miam.core.localisation.Localisation
import com.miam.core.model.Ingredient
import com.miam.core.viewModels.quantityFormatter.QuantityFormatter
import com.miam.kmm_miam_sdk.android.ressource.Image.miamDefaultIngredient
import com.miam.kmm_miam_sdk.android.theme.Colors
import com.miam.kmm_miam_sdk.android.theme.Typography
import java.util.Locale
import kotlin.math.ceil

class IngredientsImp: Ingredients {
    @Composable
    override fun Content(params: IngredientsParameters) {
        Column(Modifier.padding(vertical = 16.dp), horizontalAlignment = Alignment.CenterHorizontally) {
            Text(
                modifier = Modifier
                    .padding(horizontal = 16.dp)
                    .fillMaxWidth(),
                text = Localisation.Recipe.numberOfIngredients(params.ingredients.size).localised,
                style = Typography.subtitleBold.copy(textAlign = TextAlign.Start),
                color = Colors.black
            )

            params.ingredients.groupBy { ceil(((params.ingredients.indexOf(it) + 1.0)) / 3.0) }.map { it.value }.forEach { row ->
                Row(
                    modifier = Modifier
                        .fillMaxWidth()
                        .padding(vertical = 16.dp),
                    horizontalArrangement = Arrangement.SpaceEvenly
                ) {
                    row.forEach {
                        Box(Modifier.weight(1f), contentAlignment = Alignment.Center) {
                            Ingredient(it, params.guestsCount, params.defaultRecipeGuest)
                        }
                    }
                    for (i in 3 - row.size downTo 1) {
                        Spacer(modifier = Modifier.weight(1f))
                    }
                }
            }
        }
    }

    @Composable
    fun Ingredient(ingredient: Ingredient, guestsCount: Int, defaultRecipeGuest: Int) {
        Column(
            horizontalAlignment = Alignment.CenterHorizontally
        ) {

            Surface(
                shape = RoundedCornerShape(32.dp),
                border = BorderStroke(1.dp, Colors.lightgrey)
            ) {
                if (ingredient.attributes?.pictureUrl.isNullOrEmpty()) {
                    Image(
                        painter = painterResource(miamDefaultIngredient),
                        contentDescription = "icon ingredient",
                        contentScale = ContentScale.Crop,
                        modifier = Modifier
                            .height(56.dp)
                            .width(56.dp)
                            .padding(4.dp)
                    )

                } else {
                    Image(
                        painter = rememberImagePainter(ingredient.attributes?.pictureUrl),
                        contentDescription = "icon ingredient",
                        contentScale = ContentScale.Crop,
                        modifier = Modifier
                            .height(56.dp)
                            .width(56.dp)
                            .padding(4.dp)
                    )
                }
            }
            Text(
                text = ingredient.attributes?.name?.replaceFirstChar { if (it.isLowerCase()) it.titlecase(Locale.ROOT) else it.toString() } ?: "",
                textAlign = TextAlign.Center,
                style = TextStyle(
                    fontSize = 14.sp,
                    lineHeight = 21.sp,
                    fontWeight = FontWeight(900),
                    color = Colors.boldText,
                    textAlign = TextAlign.Center,
                )
            )
            Text(
                text = QuantityFormatter.readableFloatNumber(
                    value = QuantityFormatter.realQuantities(
                        // Will never append ingredient must have a quantity
                        ingredient.attributes?.quantity ?: "1",
                        guestsCount,
                        // Will never append recipe must have a numberOfGuests
                        defaultRecipeGuest
                    ),
                    unit = ingredient.attributes?.unit
                ), textAlign = TextAlign.Center
            )
        }
    }
}