package com.miam.sdk.components.recipeDetail.success

import androidx.compose.foundation.background
import androidx.compose.foundation.layout.Box
import androidx.compose.foundation.layout.Column
import androidx.compose.foundation.layout.Row
import androidx.compose.foundation.layout.fillMaxWidth
import androidx.compose.foundation.layout.height
import androidx.compose.foundation.layout.padding
import androidx.compose.foundation.rememberScrollState
import androidx.compose.foundation.verticalScroll
import androidx.compose.material.BottomAppBar
import androidx.compose.material.Scaffold
import androidx.compose.runtime.Composable
import androidx.compose.runtime.DisposableEffect
import androidx.compose.runtime.LaunchedEffect
import androidx.compose.runtime.collectAsState
import androidx.compose.runtime.getValue
import androidx.compose.runtime.mutableStateOf
import androidx.compose.runtime.remember
import androidx.compose.runtime.setValue
import androidx.compose.ui.Modifier
import androidx.compose.ui.graphics.Color
import androidx.compose.ui.unit.dp
import com.miam.core.model.Recipe
import com.miam.core.viewModels.dynamicRecipeDetail.DynamicRecipeDetailContract
import com.miam.core.viewModels.dynamicRecipeDetail.DynamicRecipeDetailViewModel
import com.miam.core.viewModels.dynamicRecipeDetailFooter.DynamicRecipeDetailFooterContract
import com.miam.kmm_miam_sdk.android.theme.Colors
import com.miam.sdk.components.recipeDetail.success.footer.RecipeDetailSuccessFooterParameters
import com.miam.sdk.components.recipeDetail.success.header.RecipeDetailHeaderParameters
import com.miam.sdk.components.recipeDetail.success.info.RecipeDetailInfoParameters
import com.miam.sdk.components.recipeDetail.success.ingredients.IngredientsParameters
import com.miam.sdk.components.recipeDetail.success.oftenDeleted.OftenDeletedProducts
import com.miam.sdk.components.recipeDetail.success.product.Products
import com.miam.sdk.components.recipeDetail.success.sponsorBanner.RecipeDetailSponsorBannerParameters
import com.miam.sdk.components.recipeDetail.success.steps.RecipeDetailStepsParamters
import com.miam.sdk.components.recipeDetail.success.swapper.SwapperParameters
import com.miam.sdk.components.recipeDetail.success.tag.RecipeDetailSuccessTagParameters
import com.miam.sdk.components.recipeDetail.success.unavailable.UnavailableProducts
import com.miam.sdk.di.TemplateDI


@Composable
internal fun RecipeDetailSuccess(
    recipe: Recipe,
    viewModel: DynamicRecipeDetailViewModel,
    isLikeEnabled: Boolean,
    cookOnlyMode: Boolean,
    goToSponsor: (String) -> Unit
) {

    val saveViewModel by remember { mutableStateOf(viewModel) }
    val guestState = saveViewModel.recipeGuests.collectAsState()

    var isShopping by remember { mutableStateOf(!cookOnlyMode) }

    DisposableEffect(Unit) { onDispose {
        saveViewModel.dynamicRecipeDetailFooterViewModel.dispose()
    } }

    LaunchedEffect(Unit) {
        saveViewModel.dynamicRecipeDetailFooterViewModel.registerListeners()
    }

    Scaffold(
        bottomBar = {
            if (!cookOnlyMode) {
                val footerState by saveViewModel.dynamicRecipeDetailFooterViewModel.uiState.collectAsState()
                BottomAppBar(Modifier.height(60.dp), backgroundColor = Colors.white) {
                    TemplateDI.recipeDetail.success.footer.view.Content(
                        params = RecipeDetailSuccessFooterParameters(
                            price = saveViewModel.dynamicRecipeDetailFooterViewModel.pricePerGuest,
                            priceStatus = footerState.priceStatus,
                            ingredientsStatus = footerState.ingredientsStatus,
                            isButtonLock = saveViewModel.dynamicRecipeDetailFooterViewModel.isAddingRemainingIngredients,
                            onConfirm = { saveViewModel.dynamicRecipeDetailFooterViewModel.setEvent(DynamicRecipeDetailFooterContract.Event.OnTapContinueShopping) }
                        )
                    )
                }
            }
        }
    )
    { padding ->
        Box {
            Column(
                modifier = Modifier
                    .fillMaxWidth()
                    .padding(padding)
                    .verticalScroll(rememberScrollState())
            ) {
                TemplateDI.recipeDetail.success.info.view.Content(
                    params = RecipeDetailInfoParameters(
                        recipe = recipe,
                        guestCount = guestState.value,
                        isUpdating = false,
                        isLikeEnable = isLikeEnabled,
                        showGuestCounter = !cookOnlyMode,
                        { saveViewModel.setEvent(DynamicRecipeDetailContract.Event.UpdateGuests(it)) }
                    ) { saveViewModel.continueShopping() }
                )
                if (recipe.isSponsored) {
                    for (sponsor in recipe.relationships?.sponsors?.data ?: emptyList()) {
                        TemplateDI.recipeDetail.success.sponsorBanner.view.Content(
                            params = RecipeDetailSponsorBannerParameters(sponsor = sponsor)
                            { goToSponsor(it.id) }
                        )
                    }
                }
                TemplateDI.recipeDetail.success.tag.view.Content(
                    params = RecipeDetailSuccessTagParameters(
                        title = recipe.attributes?.title ?: "",
                        totalTime = recipe.totalTime,
                        preparationTime = recipe.attributes?.preparationTime,
                        cookingTime = recipe.attributes?.cookingTime,
                        restingTime = recipe.attributes?.restingTime,
                        difficulty = recipe.difficulty
                    )
                )
                if (!cookOnlyMode) {
                    TemplateDI.recipeDetail.success.swapper.view.Content(
                        params = SwapperParameters(isShopping = false) { isShopping = !isShopping }
                    )
                }
                if (isShopping) {
                    Products(
                        productsVm = saveViewModel.canBeAddedProduct,
                        defaultRecipeGuest = recipe.attributes?.numberOfGuests ?: 4,
                        guest = saveViewModel.recipeGuests
                    )
                    OftenDeletedProducts(
                        productsVm = saveViewModel.oftenDeletedProduct,
                        guestsCount = saveViewModel.recipeGuests,
                        defaultRecipeGuest = recipe.attributes?.numberOfGuests ?: 4
                    )
                    UnavailableProducts(
                        productsVm = saveViewModel.unavailableProduct,
                        defaultRecipeGuest = recipe.attributes?.numberOfGuests ?: 4,
                        guestsCount = saveViewModel.recipeGuests
                    )
                } else {
                    Column {
                        recipe.relationships?.ingredients?.data?.let { ingredients ->
                            TemplateDI.recipeDetail.success.ingredients.view.Content(
                                params = IngredientsParameters(
                                    ingredients = ingredients,
                                    guestsCount = guestState.value,
                                    defaultRecipeGuest = recipe.attributes?.numberOfGuests ?: 4
                                )
                            )
                        }
                        TemplateDI.recipeDetail.success.steps.view.Content(
                            params = RecipeDetailStepsParamters(
                                recipe.sortedStep
                            )
                        )
                    }
                }
            }
        }
        Row(
            modifier = Modifier
                .fillMaxWidth()
                .height(100.dp)
                .background(Color.Transparent)
        ) {
            TemplateDI.recipeDetail.success.header.view.Content(
                params = RecipeDetailHeaderParameters(
                    // Will never append recipe must have a title
                    title = recipe.attributes?.title ?: "",
                    closeDialogue = { saveViewModel.continueShopping() }
                )
            )
        }
    }

}