package com.miam.sdk.components.recipeDetail

import android.content.Context
import android.util.AttributeSet
import androidx.compose.runtime.Composable
import androidx.compose.runtime.DisposableEffect
import androidx.compose.runtime.LaunchedEffect
import androidx.compose.runtime.collectAsState
import androidx.compose.runtime.getValue
import androidx.compose.ui.platform.AbstractComposeView
import com.miam.core.viewModels.dynamicRecipeDetail.DynamicRecipeDetailContract
import com.miam.core.viewModels.dynamicRecipeDetail.DynamicRecipeDetailViewModel
import com.miam.sdk.components.baseComponent.ManagementResourceState
import com.miam.sdk.components.baseComponent.loader.LoaderParameters
import com.miam.sdk.components.recipeDetail.success.RecipeDetailSuccess
import com.miam.sdk.di.TemplateDI


class RecipeDetail @JvmOverloads constructor(
    context: Context,
    attrs: AttributeSet? = null,
    defStyleAttr: Int = 0
): AbstractComposeView(context, attrs, defStyleAttr) {

    private lateinit var viewModel: DynamicRecipeDetailViewModel
    private lateinit var recipeId: String
    private lateinit var goToSponsor: (sponsorId: String) -> Unit

    fun bind(
        recipeId: String,
        back: () -> Unit,
        changeProduct: (sponsorId: String) -> Unit,
        goToSponsor: (sponsorId: String) -> Unit
    ) {
        viewModel = DynamicRecipeDetailViewModel(back, changeProduct)
        this.recipeId = recipeId
        this.goToSponsor = goToSponsor
    }

    @Composable
    override fun Content() {
        View(viewModel = viewModel, recipeId = recipeId, goToSponsor = goToSponsor)
    }

    companion object {

        @Composable
        fun View(
            viewModel: DynamicRecipeDetailViewModel,
            recipeId: String,
            goToSponsor: (sponsorId: String) -> Unit,
            cookOnlyMode: Boolean = false
        ) {
            val state by viewModel.uiState.collectAsState()
            LaunchedEffect(recipeId) {
                viewModel.setEvent(DynamicRecipeDetailContract.Event.SetRecipeId(recipeId))
            }
            LaunchedEffect(Unit) {
                viewModel.registerListeners()
            }
            DisposableEffect(Unit) { onDispose {
                viewModel.dispose()
            } }

            ManagementResourceState(
                resourceState = state.recipe,
                successView = { recipe ->
                    requireNotNull(recipe)
                    RecipeDetailSuccess(
                        recipe = recipe,
                        viewModel = viewModel,
                        isLikeEnabled = state.likeIsEnable,
                        cookOnlyMode = cookOnlyMode,
                        goToSponsor = { goToSponsor(it) }
                    )
                },
                loadingView = {
                    TemplateDI.recipeDetail.loading?.view?.Content(params = LoaderParameters)
                        ?: TemplateDI.defaultViews.loading.view.Content(params = LoaderParameters)
                }
            )
        }
    }
}