package com.miam.sdk.components.mealPlanner.basketPreview.success

import androidx.compose.foundation.layout.Column
import androidx.compose.foundation.lazy.LazyColumn
import androidx.compose.foundation.lazy.itemsIndexed
import androidx.compose.runtime.Composable
import androidx.compose.runtime.DisposableEffect
import androidx.compose.runtime.LaunchedEffect
import androidx.compose.runtime.collectAsState
import androidx.compose.runtime.getValue
import androidx.compose.runtime.key
import androidx.compose.runtime.mutableStateOf
import androidx.compose.runtime.remember
import androidx.compose.runtime.setValue
import com.miam.core.localisation.Localisation
import com.miam.core.model.Recipe
import com.miam.core.viewModels.dynamicRecipeDetail.DynamicRecipeDetailContract
import com.miam.core.viewModels.mealPlanner.basketPreview.recipeRow.MealPlannerBasketPreviewRecipeRowViewModel
import com.miam.core.viewModels.mealPlanner.basketPreview.recipeRow.RecipeRowState
import com.miam.sdk.components.baseComponent.ManagementResourceState
import com.miam.sdk.components.mealPlanner.basketPreview.success.found.FoundProducts
import com.miam.sdk.components.mealPlanner.basketPreview.success.notInBasket.NotInBasketProducts
import com.miam.sdk.components.mealPlanner.basketPreview.success.recipeRow.MealPlannerBasketPreviewSuccessRecipeRowParameters
import com.miam.sdk.components.recipeJourney.RecipeJourney
import com.miam.sdk.di.TemplateDI
import kotlinx.coroutines.flow.StateFlow

@Composable
fun MealPlannerBasketPreviewSuccess(rowViewModels: StateFlow<List<MealPlannerBasketPreviewRecipeRowViewModel>>) {

    val rowState by rowViewModels.collectAsState()

    LazyColumn() {
        itemsIndexed(
            rowState,
            key = { index, recipeRow ->
                recipeRow.hashCode()
            }
        ) { _, recipeRowVM ->
            Recipe(recipeRowVM = recipeRowVM)
        }
    }
}

@Composable
fun RecipeInRecipeJourney(
    recipe: Recipe, rowState: RecipeRowState, numberOfGuest: StateFlow<Int>, isExpanded: Boolean,
    isLock: StateFlow<Boolean>,
    onGuestChange: (Int) -> Unit,
    toggleCollapse: () -> Unit,
    deleteRecipe: () -> Unit
) {
    val guestsState by numberOfGuest.collectAsState()

    RecipeJourney.View(recipe = recipe, cookOnlyMode = true) { recipeId, goToDetail ->
        TemplateDI.mealPlanner.basketPreview.success.recipe.view.Content(
            MealPlannerBasketPreviewSuccessRecipeRowParameters(
                isExpanded = isExpanded,
                isDeleting = rowState.recipeDeleting,
                isPriceRefreshing = rowState.recipePriceRefreshing,
                isChildrenLoading = rowState.childrenLoading,
                productCount = rowState.allFoundProductCount,
                isLock = isLock,
                price = rowState.recipePrice,
                picture = recipe.attributes?.mediaUrl ?: "",
                name = recipe.attributes?.title ?: "",
                numberOfGuest = guestsState,
                openRecipeDetail = {
                    goToDetail(recipeId)
                },
                onGuestChange = onGuestChange,
                toggleCollapse = toggleCollapse,
                delete = deleteRecipe
            )
        )
    }
}

@Composable
fun Recipe(recipeRowVM: MealPlannerBasketPreviewRecipeRowViewModel) {

    // All recipe state (fetching loaded ... )
    val recipeState by recipeRowVM.uiState.collectAsState()
    // All Info about row ( guess , is deleting , is price refreshing ... )
    val rowState by recipeRowVM.recipeRowState.collectAsState()

    ManagementResourceState(resourceState = recipeState.recipe, successView = { recipe ->
        requireNotNull(recipe)

        LaunchedEffect(recipe.id) {
            recipeRowVM.setEvent(DynamicRecipeDetailContract.Event.SetRecipeId(recipe.id))
            recipeRowVM.registerListeners()
        }
        DisposableEffect(Unit) {
            onDispose {
                // we need to make sure we get recipe state before disposing it so we don't miss anything
                if (!rowState.childrenLoading) {
                    recipeRowVM.dispose()
                }
            }
        }

        var expandedState by remember { mutableStateOf(false) }

        Column {
            RecipeInRecipeJourney(
                recipe,
                rowState,
                recipeRowVM.recipeGuests,
                expandedState,
                isLock = recipeRowVM.guestIsUpdating,
                { newGuestCount -> recipeRowVM.setEvent(DynamicRecipeDetailContract.Event.UpdateGuests(newGuestCount)) },
                { expandedState = !expandedState },
                { recipeRowVM.deleteRecipe() }
            )

            if (expandedState) {
                val oftenDeleted by recipeRowVM.oftenDeletedProduct.collectAsState()
                val deleted by recipeRowVM.deletedProduct.collectAsState()
                val unavailable by recipeRowVM.unavailableProduct.collectAsState()

                FoundProducts(recipeRowVM.foundProduct)
                if (oftenDeleted.isNotEmpty()) {
                    NotInBasketProducts(
                        headerTitle = Localisation.Basket.ownedProducts(oftenDeleted.size).localised,
                        notInBasketProducts = recipeRowVM.oftenDeletedProduct
                    )
                }
                if (unavailable.isNotEmpty()) {
                    NotInBasketProducts(
                        headerTitle = Localisation.Basket.unavailableProducts(unavailable.size).localised,
                        notInBasketProducts = recipeRowVM.unavailableProduct
                    )
                }
                if (deleted.isNotEmpty()) {
                    NotInBasketProducts(
                        headerTitle = Localisation.Basket.removedProducts(deleted.size).localised,
                        notInBasketProducts = recipeRowVM.deletedProduct
                    )
                }
            }
        }
    })
}