package com.miam.kmm_miam_sdk.android.ui.components.counter

import androidx.compose.foundation.Image
import androidx.compose.foundation.background
import androidx.compose.foundation.layout.Arrangement
import androidx.compose.foundation.layout.Row
import androidx.compose.foundation.layout.height
import androidx.compose.foundation.layout.padding
import androidx.compose.foundation.layout.size
import androidx.compose.foundation.shape.RoundedCornerShape
import androidx.compose.material.CircularProgressIndicator
import androidx.compose.material.IconButton
import androidx.compose.material.Text
import androidx.compose.runtime.Composable
import androidx.compose.runtime.getValue
import androidx.compose.runtime.mutableStateOf
import androidx.compose.runtime.remember
import androidx.compose.runtime.setValue
import androidx.compose.ui.Alignment
import androidx.compose.ui.Modifier
import androidx.compose.ui.draw.clip
import androidx.compose.ui.graphics.ColorFilter
import androidx.compose.ui.res.painterResource
import androidx.compose.ui.text.TextStyle
import androidx.compose.ui.text.font.FontWeight
import androidx.compose.ui.tooling.preview.Preview
import androidx.compose.ui.unit.dp
import com.miam.kmm_miam_sdk.android.theme.Colors
import com.miam.kmm_miam_sdk.android.theme.Dimension
import com.miam.kmm_miam_sdk.android.ui.components.counter.CounterImage.guests
import com.miam.kmm_miam_sdk.android.ui.components.counter.CounterImage.less
import com.miam.kmm_miam_sdk.android.ui.components.counter.CounterImage.plus
import com.miam.kmm_miam_sdk.android.ui.components.counter.CounterStyle.countText
import com.miam.kmm_miam_sdk.android.ui.components.counter.CounterStyle.guestIcon
import com.miam.kmm_miam_sdk.android.ui.components.counter.CounterStyle.guestIconLight


@Composable
fun Counter(
    initialCount: Int?,
    isDisable: Boolean = false,
    onCounterChanged: (newValue: Int) -> Unit,
    lightMode: Boolean = false,
    minValue: Int? = null,
    maxValue: Int? = null,
    isLoading: Boolean = false,
    key: Any? = null
) {
    var localCount by remember(key ?: initialCount) { mutableStateOf(initialCount) }

    fun newValueBounded(newValue: Int): Boolean {
        return (minValue == null || newValue >= minValue) && (maxValue == null || newValue <= maxValue)
    }

    fun changedValue(localCount: Int?, delta: Int): Int? {
        // if min value is not defined 1 is surely bounded
        if (localCount == null) return minValue ?: 1

        if (!newValueBounded(localCount + delta)) return null

        return localCount + delta
    }

    fun increase() {
        changedValue(localCount, 1)?.let { newCount ->
            localCount = newCount
            onCounterChanged(newCount)
        }
    }

    fun decrease() {
        changedValue(localCount, -1)?.let { newCount ->
            localCount = newCount
            onCounterChanged(newCount)
        }
    }


    Row(
        modifier = Modifier.padding(
            horizontal = Dimension.sPadding,
            vertical = Dimension.sPadding
        ),
        verticalAlignment = Alignment.CenterVertically,
    ) {
        Image(
            painter = painterResource(guests),
            contentDescription = "guests icon",
            modifier = if (lightMode) guestIconLight else guestIcon
        )
        Row(
            verticalAlignment = Alignment.CenterVertically,
        ) {
            Plus({ decrease() }, isDisable)
            MiddleText(localCount, lightMode, isLoading)
            Minus({ increase() }, isDisable)
        }
    }
}

@Composable
fun Plus(decrease: () -> Unit, isDisable: Boolean) {
    IconButton(
        onClick = { decrease() },
        enabled = !isDisable,
        modifier = Modifier
            .clip(RoundedCornerShape(topStart = 70f, bottomStart = 70f))
            .background(Colors.white)
    ) {
        Image(
            painter = painterResource(less),
            contentDescription = "less icon",
            colorFilter = ColorFilter.tint(Colors.primary),
            modifier = Modifier.size(Dimension.mIconHeight)
        )
    }
}

@Composable
fun MiddleText(localCount: Int?, lightMode: Boolean, isLoading: Boolean) {
    fun counterText(countValue: Int?, lightMode: Boolean): String {
        if (countValue == null) return "-"
        return localCount.toString()
    }

    Row(
        modifier = Modifier
            .background(color = Colors.white)
            .height(48.dp),
        verticalAlignment = Alignment.CenterVertically,
        horizontalArrangement = Arrangement.Center,
    ) {
        if (isLoading) {
            CircularProgressIndicator(color = Colors.boldText,modifier= Modifier.size(16.dp))
        } else {
            Row(verticalAlignment = Alignment.CenterVertically) {
                Text(
                    text = counterText(localCount, lightMode),
                    color = Colors.boldText,
                    style = TextStyle(fontWeight = FontWeight.Bold),
                    modifier = countText
                )
                if (!lightMode) {
                    Image(
                        painter = painterResource(guests),
                        contentDescription = "guests icon",
                        modifier = Modifier.size(16.dp)
                    )
                }
            }
        }
    }
}

@Composable
fun Minus(increase: () -> Unit, isDisable: Boolean) {
    IconButton(
        onClick = { increase() },
        enabled = !isDisable,
        modifier = Modifier
            .clip(RoundedCornerShape(topEnd = 70f, bottomEnd = 70f))
            .background(Colors.white)
    ) {
        Image(
            painter = painterResource(plus),
            contentDescription = null,
            colorFilter = ColorFilter.tint(Colors.primary),
            modifier = Modifier.size(Dimension.mIconHeight)
        )
    }
}


@Preview
@Composable
fun CounterPreview() {
    Counter(12, false, {}, false)
}

@Preview
@Composable
fun lightCounterPreview() {
    Counter(12, false, {}, true)
}