package com.miam.sdk.components.mealPlanner.form

import android.content.Context
import android.util.AttributeSet
import androidx.compose.runtime.Composable
import androidx.compose.runtime.collectAsState
import androidx.compose.runtime.getValue
import androidx.compose.runtime.rememberCoroutineScope
import androidx.compose.ui.platform.AbstractComposeView
import androidx.lifecycle.Lifecycle
import com.miam.core.sdk.base.state.ComponentUiState
import com.miam.core.sdk.viewModels.mealPlanner.form.MealPlannerFormViewModel
import com.miam.kmmMiamCore.handler.LogHandler
import com.miam.sdk.components.baseComponent.lifeCycle.OnLifecycleEvent
import com.miam.sdk.di.TemplateDI
import com.miam.sdk.templateParameters.mealPlanner.form.MealPlannerFormParameters
import kotlinx.coroutines.CoroutineExceptionHandler
import kotlinx.coroutines.launch

class MealPlannerForm @JvmOverloads constructor(
    context: Context,
    attrs: AttributeSet? = null,
    defStyleAttr: Int = 0
): AbstractComposeView(context, attrs, defStyleAttr) {

    private val formViewModel = MealPlannerFormViewModel()
    private var onRecipeFounded: (recipesId: List<String>) -> Unit = {}

    private val coroutineHandler = CoroutineExceptionHandler { _, exception ->
        LogHandler.error("Miam error in MealPlannerForm $exception")
    }
    init {
        formViewModel.onInitComponent()
    }
    fun bind(onRecipeFounded: (recipesId: List<String>) -> Unit = {}) {
        this.onRecipeFounded = onRecipeFounded
    }

    private suspend fun loadRecipe(budget: Int, numberOfMeals: Int, numberOfGuests: Int) {
        try {
            val recipes = formViewModel.getRecipesForBudgetConstraint(budget, numberOfMeals, numberOfGuests)
            if (formViewModel.currentState.uiState != ComponentUiState.SUCCESS) return
            onRecipeFounded(recipes)
        } catch (e: Throwable) {
            formViewModel.getRecipesMaxCountForBudgetConstraint(budget, numberOfMeals)
        }
    }

    @Composable
    override fun Content() {
        val state by formViewModel.uiState.collectAsState()
        val scope = rememberCoroutineScope()

        OnLifecycleEvent { _, event ->
            when (event) {
                Lifecycle.Event.ON_CREATE -> {
                    formViewModel.getRecipesMaxCountForBudgetConstraint(state.budget, state.numberOfMeals)
                }

                else -> {
                    return@OnLifecycleEvent
                }
            }
        }

        TemplateDI.mealPlannerForm.view.Content(
            mealPlannerFormParameters = MealPlannerFormParameters(
                state.budget,
                state.numberOfGuests,
                state.numberOfMeals,
                state.uiState,
                {
                    formViewModel.setBudget(it)
                },
                {
                    formViewModel.setNumberOfGuests(it)
                },
                {
                    formViewModel.setNumberOfMeals(it)
                },
                state.recipeMaxCount,
                formViewModel.recipeCountFlow,
                { budget, numberOfGuests ->
                    formViewModel.getRecipesMaxCountForBudgetConstraint(budget, numberOfGuests)
                },
                { budget, numberOfMeals, numberOfGuests ->
                    scope.launch(coroutineHandler) {
                        loadRecipe(budget, numberOfMeals, numberOfGuests)
                    }
                }
            )
        )
    }
}