package com.miam.sdk.components.baseComponent.lifeCycle

import androidx.compose.foundation.ScrollState
import androidx.compose.foundation.lazy.LazyListState
import androidx.compose.foundation.lazy.grid.LazyGridState
import androidx.compose.ui.geometry.Rect
import androidx.compose.ui.layout.LayoutCoordinates
import androidx.compose.ui.layout.boundsInParent
import androidx.compose.ui.unit.Density

// Generic Interface
interface ListVisibilityInfo {
    val viewportStartOffset: Float
    val viewportEndOffset: Float
}

// Implementations
class LazyListVisibilityInfo(private val lazyListState: LazyListState) : ListVisibilityInfo {
    override val viewportStartOffset: Float
        get() = lazyListState.layoutInfo.viewportStartOffset.toFloat()
    override val viewportEndOffset: Float
        get() = lazyListState.layoutInfo.viewportEndOffset.toFloat()
}

class LazyGridVisibilityInfo(private val lazyGridState: LazyGridState) : ListVisibilityInfo {
    override val viewportStartOffset: Float
        get() = lazyGridState.layoutInfo.viewportStartOffset.toFloat()
    override val viewportEndOffset: Float
        get() = lazyGridState.layoutInfo.viewportEndOffset.toFloat()
}

class ScrollableListVisibilityInfo(
    private val scrollState: ScrollState,
    private val density: Density
) : ListVisibilityInfo {
    override val viewportStartOffset: Float
        get() = with(density) { scrollState.value.toFloat().toDp().value }
    override val viewportEndOffset: Float
        get() = with(density) { (scrollState.value + scrollState.maxValue).toFloat().toDp().value }
}

// Function to determine if view is seen
fun checkIfVisible(
    coordinates: LayoutCoordinates,
    visibilityInfo: ListVisibilityInfo
): Boolean {
    val visibleArea = Rect(
        left = 0f,
        top = visibilityInfo.viewportStartOffset,
        right = visibilityInfo.viewportEndOffset,
        bottom = visibilityInfo.viewportEndOffset
    )

    val itemBounds = coordinates.boundsInParent()
    return visibleArea.overlaps(itemBounds)
}
