package com.miam.kmm_miam_sdk.android.ui.components.recipeDetails


import androidx.compose.foundation.Image
import androidx.compose.foundation.layout.Arrangement
import androidx.compose.foundation.layout.Column
import androidx.compose.foundation.layout.Row
import androidx.compose.foundation.layout.Spacer
import androidx.compose.foundation.layout.padding
import androidx.compose.foundation.rememberScrollState
import androidx.compose.foundation.verticalScroll
import androidx.compose.material.BottomAppBar
import androidx.compose.material.Divider
import androidx.compose.material.Scaffold
import androidx.compose.material.Text
import androidx.compose.runtime.Composable
import androidx.compose.runtime.collectAsState
import androidx.compose.runtime.getValue
import androidx.compose.ui.Alignment
import androidx.compose.ui.Modifier
import androidx.compose.ui.graphics.RectangleShape
import androidx.compose.ui.layout.ContentScale
import androidx.compose.ui.text.style.TextAlign
import androidx.compose.ui.tooling.preview.Preview
import androidx.compose.ui.unit.dp
import coil.compose.rememberImagePainter
import com.miam.kmmMiamCore.component.recipe.RecipeContract
import com.miam.kmmMiamCore.component.recipe.RecipeViewModel
import com.miam.kmmMiamCore.component.router.RouterOutletContract
import com.miam.kmmMiamCore.component.router.RouterOutletViewModel
import com.miam.kmmMiamCore.miam_core.model.Recipe
import com.miam.kmmMiamCore.miam_core.model.Sponsor
import com.miam.kmmMiamCore.miam_core.model.fake.RecipeFakeFactory
import com.miam.kmm_miam_sdk.android.theme.Template
import com.miam.kmm_miam_sdk.android.theme.Template.recipeDetailIngredientTemplate
import com.miam.kmm_miam_sdk.android.theme.Template.recipeDetailStepsTemplate
import com.miam.kmm_miam_sdk.android.theme.Typography.subtitleBold
import com.miam.kmm_miam_sdk.android.ui.components.likeButton.LikeButton
import com.miam.kmm_miam_sdk.android.ui.components.price.RecipePrice
import com.miam.kmm_miam_sdk.android.ui.components.recipeDetails.RecipeDetailsColor.footerSectionBackgroundColor
import com.miam.kmm_miam_sdk.android.ui.components.recipeDetails.RecipeDetailsStyle.mainColumnsContainer
import com.miam.kmm_miam_sdk.android.ui.components.recipeDetails.RecipeDetailsStyle.recipeDetailsActionsContainer
import com.miam.kmm_miam_sdk.android.ui.components.recipeDetails.RecipeDetailsStyle.recipeImageModifier
import com.miam.kmm_miam_sdk.android.ui.components.recipeDetails.RecipeDetailsStyle.titleModifier
import com.miam.kmm_miam_sdk.android.ui.components.recipeDetails.subComponents.RecipeDetailFooter
import com.miam.kmm_miam_sdk.android.ui.components.recipeDetails.subComponents.RecipeDetailSponsorBanner
import com.miam.kmm_miam_sdk.android.ui.components.recipeDetails.subComponents.RecipeDetailSteps
import com.miam.kmm_miam_sdk.android.ui.components.recipeDetails.subComponents.RecipeDetailsHeader
import com.miam.kmm_miam_sdk.android.ui.components.recipeDetails.subComponents.RecipeDifficultyAndTiming
import com.miam.kmm_miam_sdk.android.ui.components.recipeDetails.subComponents.RecipeIngredients
import com.miam.kmm_miam_sdk.android.ui.components.routerOutlet.RouterOutlet
import com.miam.kmm_miam_sdk.android.ui.components.states.ManagementResourceState


@Composable
fun RecipeDetails(
    vmRecipeCard: RecipeViewModel,
    vmRouter: RouterOutletViewModel,
    closeDialogue: () -> Unit,
    previous: () -> Unit, // TODO : To be used in v4
) {

    val state by vmRecipeCard.uiState.collectAsState()

    ManagementResourceState(
        resourceState = state.recipeState,
        successView = { recipe ->
            requireNotNull(recipe)
            RecipeDetailContent(
                recipe,
                vmRecipeCard,
                vmRouter,
                closeDialogue,
                vmRouter.currentState.showFooter,
                vmRouter.currentState.showGuestCounter
            )
        },
        loadingView = {
            Template.recipeDetailsLoader?.let { it() }
                ?: RecipeDetailsLoading()
        }
    )
}

@Composable
private fun RecipeDetailContent(
    recipe: Recipe,
    vmRecipeCard: RecipeViewModel,
    vmRouter: RouterOutletViewModel,
    closeDialogue: () -> Unit,
    withBottomBar: Boolean = true,
    showGuestCounter: Boolean = true
) {
    val scrollState = rememberScrollState()

    fun seeProductMatching() {
        vmRouter.setEvent(
            RouterOutletContract.Event.GoToPreview(
                recipeId = recipe.id,
                vm = vmRecipeCard
            )
        )
    }

    fun buy() {
        vmRecipeCard.setEvent(RecipeContract.Event.OnAddRecipe)
        vmRouter.setEvent(
            RouterOutletContract.Event.GoToPreview(
                recipeId = recipe.id,
                vm = vmRecipeCard
            )
        )
    }

    fun seeSponsorDetail(sponsor: Sponsor) {
        vmRouter.setEvent(
            RouterOutletContract.Event.GoToSponsor(
                sponsor = sponsor
            )
        )
    }

    Scaffold(
        topBar = {
            if (Template.recipeDetailHeaderTemplate != null) {
                Template.recipeDetailHeaderTemplate?.let { it({ closeDialogue() }, recipe) }
            } else {
                RecipeDetailsHeader(recipe.attributes!!.title, scrollState.value) { closeDialogue() }
            }

        },
        content =
        { padding ->
            Column(
                modifier = mainColumnsContainer
                    .padding(padding)
                    .verticalScroll(scrollState)
            ) {
                if (recipe.isSponsored) {
                    for (sponsor in recipe.relationships?.sponsors?.data ?: emptyList()) {
                        RecipeDetailSponsorBanner(sponsor = sponsor) { seeSponsorDetail(it) }
                    }
                }
                if (Template.recipeDetailInfosTemplate != null) {
                    Template.recipeDetailInfosTemplate!!({ closeDialogue() }, recipe)
                } else {
                    Image(
                        painter = rememberImagePainter(recipe.attributes!!.mediaUrl),
                        contentDescription = null,
                        contentScale = ContentScale.Crop,
                        modifier = recipeImageModifier
                    )
                    ActionRow(
                        vmRecipeCard.currentState.likeIsEnable,
                        recipe.id
                    )
                    Divider()
                    Text(
                        text = recipe.attributes!!.title,
                        modifier = titleModifier,
                        textAlign = TextAlign.Left,
                        style = subtitleBold
                    )
                    recipe.attributes?.let { attributes ->
                        RecipeDifficultyAndTiming(
                            attributes.difficulty ?: 1,
                            recipe.totalTime,
                            attributes.preparationTime,
                            attributes.cookingTime,
                            attributes.restingTime
                        )
                    }
                }

                if (recipeDetailIngredientTemplate != null) {
                    recipeDetailIngredientTemplate!!(recipe, vmRecipeCard)
                } else {
                    RecipeIngredients(recipe, vmRecipeCard, showGuestCounter)
                }
                if (recipeDetailStepsTemplate != null) {
                    recipeDetailStepsTemplate!!(
                        recipe.sortedStep,
                        vmRecipeCard
                    )
                } else {
                    RecipeDetailSteps(
                        recipe.sortedStep,
                        vmRecipeCard.currentState.activeStep
                    ) { currentStep: Int ->
                        vmRecipeCard.setEvent(RecipeContract.Event.SetActiveStep(currentStep))
                    }
                    Spacer(modifier = Modifier.padding(vertical = 50.dp))
                }

            }
        },
        bottomBar = {
            if (withBottomBar) {
                if (vmRouter.fromMealPlanner) {
                    RecipePrice(recipe.id, vmRecipeCard.currentState.guest, true)
                } else {
                    BottomAppBar(backgroundColor = footerSectionBackgroundColor) {
                        if (Template.recipeDetailFooterTemplate != null) {
                            Template.recipeDetailFooterTemplate!!(
                                recipe,
                                vmRecipeCard,
                                { seeProductMatching() },
                                { buy() }
                            )
                        } else {
                            RecipeDetailFooter(recipe.id,
                                vmRecipeCard.currentState.guest,
                                vmRecipeCard.currentState.isInCart,
                                { seeProductMatching() },
                                { buy() }
                            )
                        }
                    }
                }
            }
        }
    )
}


@Composable
fun ActionRow(likeIsEnable: Boolean, recipeId: String) {
    if (likeIsEnable) {
        Row(
            horizontalArrangement = Arrangement.SpaceBetween,
            verticalAlignment = Alignment.CenterVertically,
            modifier = recipeDetailsActionsContainer
        ) {
            val likeButton = LikeButton(RectangleShape)
            likeButton.bind(recipeId)
            likeButton.Content()
        }
    }
}

@Preview
@Composable
fun RecipeDetailContentPreview() {
    val router = RouterOutlet()
    RecipeDetailContent(RecipeFakeFactory.create(), RecipeViewModel(routerVM = router.getViewModel()), router.getViewModel(), {}, false)
}


