package com.miam.kmm_miam_sdk.android.ui.components.catalog

import androidx.compose.foundation.layout.Arrangement
import androidx.compose.foundation.layout.Column
import androidx.compose.foundation.layout.Row
import androidx.compose.foundation.layout.fillMaxWidth
import androidx.compose.foundation.layout.padding
import androidx.compose.foundation.lazy.LazyColumn
import androidx.compose.foundation.lazy.items
import androidx.compose.runtime.Composable
import androidx.compose.ui.Modifier
import androidx.compose.ui.platform.LocalContext
import androidx.compose.ui.unit.dp
import com.miam.kmmMiamCore.component.catalog.CatalogContent
import com.miam.kmmMiamCore.component.catalog.CatalogViewModel
import com.miam.kmmMiamCore.component.singletonFilter.FilterViewModelInstance
import com.miam.kmmMiamCore.miam_core.model.Package
import com.miam.kmmMiamCore.services.RouteServiceInstance
import com.miam.kmm_miam_sdk.android.theme.Template
import com.miam.kmm_miam_sdk.android.ui.components.catalog.customization.CatalogText.favoriteTitle
import com.miam.kmm_miam_sdk.android.ui.components.catalog.customization.CatalogText.filterSearchTitle
import com.miam.kmm_miam_sdk.android.ui.components.catalog.customization.CatalogText.searchPrefix
import com.miam.kmm_miam_sdk.android.ui.components.categoryRecipesPage.CategoryRecipesPage
import com.miam.kmm_miam_sdk.android.ui.components.searchRecipesPage.SearchRecipesPage


@Composable
fun CatalogSuccessView(
    categories: List<Package>,
    content: CatalogContent,
    enableFilters: Boolean,
    enablePreferences: Boolean,
    columns: Int,
    vSpacing: Int,
    hSpacing: Int,
    vmCatalog: CatalogViewModel,
    goToMealPlanner: (@Composable () -> Unit)? = null
) {

    val routeService = RouteServiceInstance.instance
    val filterVM = FilterViewModelInstance.instance

    fun back() = routeService.previous()
    val catId = vmCatalog.currentState.openedCategoryId
    val catTitle = vmCatalog.currentState.openedCategoryTitle
    val catSubtitle = vmCatalog.currentState.subtitle


    Column {
        CatalogHeader(content, enableFilters, enablePreferences, vmCatalog)
        when (content) {
            CatalogContent.CATEGORIES_LIST -> CatalogBodyCategories(categories = categories, vmCatalog = vmCatalog, goToMealPlanner)
            CatalogContent.FILTER_SEARCH -> CatalogBodyFilterSearch(::back, columns, vSpacing, hSpacing)
            CatalogContent.WORD_SEARCH -> CatalogBodyWordSearch(filterVM.currentState.searchString, columns, vSpacing, hSpacing)
            CatalogContent.CATEGORY -> CatalogBodyCategory(catId, catTitle, catSubtitle ?: "", columns, vSpacing, hSpacing)
            CatalogContent.FAVORITE -> CatalogBodyFavorite(::back, columns, vSpacing, hSpacing)
        }
    }
}

@Composable
fun CatalogBodyCategories(categories: List<Package>, vmCatalog: CatalogViewModel, goToMealPlanner: (@Composable () -> Unit)? = null) {

    if (categories.isNotEmpty()) {
        LazyColumn {
            goToMealPlanner?.let {
                item {
                    Row(
                        Modifier
                            .padding(vertical = 8.dp)
                            .fillMaxWidth(),
                        horizontalArrangement = Arrangement.Center
                    ) {
                        it()
                    }
                }
            }
            items(items = categories) { category ->
                CatalogCategory(category, LocalContext.current) {
                    vmCatalog.goToCategory(
                        category.id,
                        category.attributes?.title ?: "",
                        category.attributes?.settings?.subtitle
                    )
                }
            }
        }
    } else {
        CatalogCategoriesEmptyView {}
    }
    Template.CatalogFloatingElementTemplate?.let { it() }
}

@Composable
fun CatalogBodyFilterSearch(back: () -> Unit, columns: Int, vSpacing: Int, hSpacing: Int) {
    CatalogPage(LocalContext.current).apply { bind(filterSearchTitle, back, columns, vSpacing, hSpacing) }.Content()
}

@Composable
fun CatalogBodyWordSearch(searchStr: String?, columns: Int, vSpacing: Int, hSpacing: Int) {
    val title = "$searchPrefix \"$searchStr\""
    SearchRecipesPage(LocalContext.current).apply { bind(title, columns, vSpacing, hSpacing) }.Content()
}

@Composable
fun CatalogBodyCategory(openedCategoryId: String, openedCategoryTitle: String, subtitle: String, columns: Int, vSpacing: Int, hSpacing: Int) {
    CategoryRecipesPage(LocalContext.current).apply { bind(openedCategoryId, openedCategoryTitle, subtitle, columns, vSpacing, hSpacing) }.Content()
}

@Composable
fun CatalogBodyFavorite(back: () -> Unit, columns: Int, vSpacing: Int, hSpacing: Int) {
    // TODO use stand alone favorite
    CatalogPage(LocalContext.current).apply { bind(favoriteTitle, back, columns, vSpacing, hSpacing) }.Content()
}