package com.miam.sdk.components.mealPlanner.meals

import android.content.Context
import android.util.AttributeSet
import androidx.compose.animation.AnimatedVisibility
import androidx.compose.animation.core.tween
import androidx.compose.animation.fadeIn
import androidx.compose.animation.fadeOut
import androidx.compose.animation.slideInHorizontally
import androidx.compose.foundation.layout.Box
import androidx.compose.foundation.layout.Column
import androidx.compose.foundation.layout.padding
import androidx.compose.foundation.rememberScrollState
import androidx.compose.foundation.verticalScroll
import androidx.compose.material.ExperimentalMaterialApi
import androidx.compose.material.Scaffold
import androidx.compose.material.pullrefresh.PullRefreshIndicator
import androidx.compose.material.pullrefresh.pullRefresh
import androidx.compose.material.pullrefresh.rememberPullRefreshState
import androidx.compose.runtime.*
import androidx.compose.ui.Alignment
import androidx.compose.ui.Modifier
import androidx.compose.ui.graphics.Color
import androidx.compose.ui.platform.AbstractComposeView
import androidx.compose.ui.zIndex
import com.miam.core.sdk.data.repository.BudgetRepository
import com.miam.core.sdk.viewModels.mealPlanner.meals.MealPlannerMealsViewModel
import com.miam.kmmMiamCore.handler.LogHandler
import com.miam.kmmMiamCore.miam_core.model.MealPlannerRecipe
import com.miam.kmm_miam_sdk.android.ui.components.states.ManagementResourceState
import com.miam.sdk.components.mealPlanner.MealPlannerRecipeCard
import com.miam.sdk.di.TemplateDI
import com.miam.sdk.templateParameters.mealPlanner.planner.BudgetPlannerEmptyParameters
import com.miam.sdk.templateParameters.mealPlanner.planner.BudgetPlannerFooterParameters
import com.miam.sdk.templateParameters.mealPlanner.planner.BudgetPlannerLoadingParameters
import com.miam.sdk.templateParameters.mealPlanner.recipe.MealPlannerRecipePlaceholderParameters
import kotlinx.coroutines.CoroutineExceptionHandler
import kotlinx.coroutines.launch

class MealPlanner @JvmOverloads constructor(
    context: Context,
    attrs: AttributeSet? = null,
    defStyleAttr: Int = 0
): AbstractComposeView(context, attrs, defStyleAttr) {

    private val mealPlannerToolbar = MealPlannerToolbar(context)
    private val mealsViewModel = MealPlannerMealsViewModel()
    var goToRecipeSelector = { _: Int -> }
    var onComfirm = {}

    private val coroutineHandler = CoroutineExceptionHandler { _, exception ->
        LogHandler.error("Miam error in MealPlanner $exception")
    }

    @OptIn(ExperimentalMaterialApi::class)
    @Composable
    override fun Content() {
        DisposableEffect(Unit) { onDispose { mealsViewModel.dispose() } }

        val refreshScope = rememberCoroutineScope()
        var refreshing by remember { mutableStateOf(false) }

        val pullState = rememberPullRefreshState(refreshing, {
            refreshScope.launch(coroutineHandler) {
                refreshing = true
                mealsViewModel.onPullToRefresh()
                refreshing = false
            }
        })

        Scaffold(
            backgroundColor = Color.Transparent,
            bottomBar = {
                footerSection()
            }
        ) {
            Column(
                Modifier
                    .padding(it)
                    .pullRefresh(pullState)
                    .verticalScroll(rememberScrollState()),
                horizontalAlignment = Alignment.CenterHorizontally
            ) {

                Box(Modifier.zIndex(10f)) {
                    mealPlannerToolbar.Content()
                }
                PullRefreshIndicator(
                    refreshing, pullState,
                    Modifier
                        .zIndex(9f)
                )
                Meals()
            }
        }
    }

    @Composable
    fun Meals() {
        val state by mealsViewModel.uiState.collectAsState()
        ManagementResourceState(
            resourceState = state.meals,
            successView = { meals ->
                requireNotNull(meals)
                meals.forEachIndexed { index, recipe ->
                    MealPlannerCardSection(recipe, index, context,
                        {
                            mealsViewModel.calculAvailableBudgetOnNavigateToReplaceRecipe(state.totalPrice, recipe?.price)
                            goToRecipeSelector(it)
                        }
                    ) { mealsViewModel.removeRecipe(index) }
                }
            },
            loadingView = {
                TemplateDI.mealPlanner.plannerLoading.view.Content(
                    BudgetPlannerLoadingParameters(state.numberOfMeal)
                )
            },
            emptyView = {
                TemplateDI.mealPlanner.plannerEmpty.view.Content(
                    BudgetPlannerEmptyParameters(
                        BudgetRepository.budget,
                        BudgetRepository.guestCount,
                        BudgetRepository.mealCount,
                    )
                )
            },
            errorView = {
            }
        )
    }

    @Composable

    fun MealPlannerCardSection(
        recipe: MealPlannerRecipe?,
        index: Int,
        context: Context,
        goToChangeRecipe: (Int) -> Unit,
        removeRecipe: () -> Unit
    ) {

        val localRecipe = remember(recipe?.recipeId) { recipe?.recipeId }
        val recipeCard = remember(localRecipe) {
            localRecipe?.let {
                MealPlannerRecipeCard(context).apply {
                    idRecipe = it
                    price = recipe?.price ?: 0.0
                    changeAction = {
                        goToChangeRecipe(index)
                    }
                    deleteAction = removeRecipe
                }
            }
        }
        AnimatedVisibility(
            visible = recipe != null,
            enter = fadeIn(
                animationSpec = tween(delayMillis = (100 * index))
            ),
            exit = fadeOut(
                animationSpec = tween(durationMillis = 0)
            )

        ) {
            recipeCard?.Content()
        }


        AnimatedVisibility(
            visible = recipe == null,
            enter = slideInHorizontally(),
            exit = fadeOut(
                animationSpec = tween(durationMillis = 0)
            )

        ) {
            TemplateDI.mealPlannerRecipePlaceholder.view.Content(
                mealPlannerRecipePlaceholderParameters = MealPlannerRecipePlaceholderParameters {
                    goToChangeRecipe(index)
                }
            )
        }
    }


    @Composable
    fun footerSection() {
        val state by mealsViewModel.uiState.collectAsState()
        TemplateDI.mealPlanner.plannerFooter.view.Content(budgetPlannerFooterParameters =
        BudgetPlannerFooterParameters(
            BudgetRepository.budget,
            state.totalPrice,
            BudgetRepository.mealCount
        ) {
            onComfirm()
            mealsViewModel.addRecipesToGroceriesList()
        }
        )
    }
}

