package com.miam.sdk.components.mealPlanner.basketPreview

import android.content.Context
import android.util.AttributeSet
import androidx.compose.animation.AnimatedVisibility
import androidx.compose.foundation.layout.Column
import androidx.compose.foundation.layout.fillMaxSize
import androidx.compose.foundation.layout.padding
import androidx.compose.foundation.rememberScrollState
import androidx.compose.foundation.verticalScroll
import androidx.compose.material.Scaffold
import androidx.compose.runtime.Composable
import androidx.compose.runtime.collectAsState
import androidx.compose.runtime.getValue
import androidx.compose.runtime.mutableStateOf
import androidx.compose.runtime.remember
import androidx.compose.runtime.setValue
import androidx.compose.ui.Modifier
import androidx.compose.ui.graphics.Color
import androidx.compose.ui.platform.AbstractComposeView
import com.miam.core.sdk.localisation.Localisation
import com.miam.core.sdk.viewModels.mealPlanner.basketPreview.MealPlannerBasketPreviewViewModel
import com.miam.kmmMiamCore.component.basketPreview.BasketPreviewContract
import com.miam.kmmMiamCore.component.basketPreview.BasketPreviewViewModel
import com.miam.kmmMiamCore.component.recipe.RecipeViewModel
import com.miam.kmmMiamCore.miam_core.model.BasketEntry
import com.miam.kmmMiamCore.miam_core.model.BasketPreviewLine
import com.miam.kmm_miam_sdk.android.ui.components.routerOutlet.RouterOutlet
import com.miam.kmm_miam_sdk.android.ui.components.states.ManagementResourceState
import com.miam.sdk.di.TemplateDI
import com.miam.sdk.templateParameters.mealPlanner.basketPreview.MealPlannerBasketPreviewEmptyParameters
import com.miam.sdk.templateParameters.mealPlanner.basketPreview.MealPlannerBasketPreviewFooterParameters
import com.miam.sdk.templateParameters.mealPlanner.basketPreview.MealPlannerBasketPreviewLoadingParameters
import com.miam.sdk.templateParameters.mealPlanner.basketPreview.MealPlannerBasketPreviewProductParameters
import com.miam.sdk.templateParameters.mealPlanner.basketPreview.MealPlannerBasketPreviewRecipeOverviewParameters
import com.miam.sdk.templateParameters.mealPlanner.basketPreview.MealPlannerBasketPreviewSectionProductParameters
import com.miam.sdk.templateParameters.mealPlanner.basketPreview.MealPlannerBasketPreviewSectionTitleParameters

class MealPlannerBasketPreview @JvmOverloads constructor(
    context: Context,
    attrs: AttributeSet? = null,
    defStyleAttr: Int = 0
): AbstractComposeView(context, attrs, defStyleAttr) {

    private val mealPlannerBasketPreviewViewModel = MealPlannerBasketPreviewViewModel()
    internal val modal = RouterOutlet()


    var onEmptyState: () -> Unit = {}
    var onComfirmBasket: () -> Unit = {}

    @Composable
    override fun Content() {
        val state by mealPlannerBasketPreviewViewModel.uiState.collectAsState()

        Column() {
            modal.Content()
            ManagementResourceState(
                resourceState = state.lines,
                successView = { lines ->
                    requireNotNull(lines)
                    Scaffold(
                        backgroundColor = Color.Transparent,
                        topBar = {
                        },
                        bottomBar = {
                            TemplateDI.mealPlannerBasketPreviewFooter.view.Content(
                                mealPlannerBasketPreviewFooterParameters =
                                MealPlannerBasketPreviewFooterParameters(
                                    budget = mealPlannerBasketPreviewViewModel.budget,
                                    totalPrice = state.totalPrice,
                                    productCount = state.productCount,
                                    {
                                        mealPlannerBasketPreviewViewModel.onClickFinalize()
                                        onComfirmBasket()
                                    }
                                ) {

                                }
                            )
                        }
                    ) {
                        Column(
                            Modifier
                                .fillMaxSize()
                                .verticalScroll(rememberScrollState())
                                .padding(it)
                        ) {
                            lines.forEachIndexed { index, line ->
                                CollapsableRecipe(line, modal) {
                                    mealPlannerBasketPreviewViewModel.removeRecipe(index, lines)
                                }
                            }
                        }
                    }
                },
                loadingView = {
                    TemplateDI.mealPlannerBasketPreviewLoading.view.Content(mealPlannerBasketPreviewLoadingParameters = MealPlannerBasketPreviewLoadingParameters())
                },
                emptyView = {
                    TemplateDI.mealPlannerBasketPreviewEmpty.view.Content(mealPlannerBasketPreviewEmptyParameters = MealPlannerBasketPreviewEmptyParameters {
                        onEmptyState()
                    })
                }
            )
        }
    }

    @Composable
    private fun CollapsableRecipe(line: BasketPreviewLine, modal: RouterOutlet, removeRecipe: () -> Unit) {

        var expandedState by remember { mutableStateOf(false || line.isDeleting) }
        val recipeVM by remember {
            mutableStateOf(RecipeViewModel(modal.getViewModel()).apply {
                fetchRecipe(line.id ?: "")
            })
        }
        val vmBasketPreview by remember { mutableStateOf(BasketPreviewViewModel(line.id ?: "")) }

        TemplateDI.mealPlannerBasketPreviewRecipeOverview.view.Content(
            mealPlannerBasketPreviewRecipeOverviewParameters = MealPlannerBasketPreviewRecipeOverviewParameters(
                line.id ?: "",
                line.picture,
                line.title,
                line.entries?.found?.size ?: 0,
                line.count,
                line.price.toDouble(),
                false,
                expandedState,
                line.isDeleting,
                {},
                { expandedState = !expandedState },
                { recipeVM.goToDetail(false) },
                { guestCount -> vmBasketPreview.updateGuest({ recipeVM.updateGuest(guestCount) }, guestCount) },
                removeRecipe
            ) {
                AnimatedVisibility(visible = expandedState) {
                    Column() {
                        line.entries?.found?.map { entry -> BasketPreviewLine.fromBasketEntry(entry) }
                            ?.forEach { basketPreviewLines ->
                                TemplateDI.mealPlannerBasketPreviewProduct.view.Content(
                                    mealPlannerBasketPreviewProductParameters =
                                    MealPlannerBasketPreviewProductParameters(
                                        id = basketPreviewLines.id ?: "",
                                        picture = basketPreviewLines.picture,
                                        name = basketPreviewLines.title,
                                        description = basketPreviewLines.bplDescription[0],
                                        price = basketPreviewLines.price.toDouble(),
                                        sharedRecipeCount = basketPreviewLines.recipesCount,
                                        isSubstitutable = (basketPreviewLines.record as BasketEntry).isSubstitutable,
                                        isLoading = vmBasketPreview.isCurrentlyInUpdate(basketPreviewLines.id ?: ""),
                                        quantity = basketPreviewLines.count,
                                        onQuantityChanged = { newQuantity ->
                                            if (newQuantity == 0) {
                                                vmBasketPreview.removeBasketEntry(basketPreviewLines.record as BasketEntry)
                                            } else {
                                                vmBasketPreview.updateBasketEntry(basketPreviewLines.record as BasketEntry, newQuantity)
                                            }
                                        },
                                        delete = { vmBasketPreview.removeBasketEntry(basketPreviewLines.record as BasketEntry) },
                                        changeProduct = {
                                            vmBasketPreview.setEvent(BasketPreviewContract.Event.OpenItemSelector(basketPreviewLines))
                                            modal.goToReplaceItem()
                                        }
                                    )
                                )
                            }
                        line.entries?.let {
                            if (it.oftenDeleted.isNotEmpty()) {
                                ProductSection(
                                    it.oftenDeleted,
                                    Localisation.Basket.ownedProducts(it.oftenDeleted.size).localised,
                                    true
                                ) { entry ->
                                    vmBasketPreview.setEvent(
                                        BasketPreviewContract.Event.AddEntry(entry)
                                    )
                                }
                            }
                            if (it.notFound.isNotEmpty()) {
                                ProductSection(
                                    it.notFound,
                                    Localisation.Basket.unavailableProducts(it.notFound.size).localised,
                                    false
                                ) { entry ->
                                    vmBasketPreview.setEvent(
                                        BasketPreviewContract.Event.AddEntry(entry)
                                    )
                                }
                            }
                            if (it.removed.isNotEmpty()) {
                                ProductSection(
                                    line.entries?.removed ?: emptyList(),
                                    Localisation.Basket.removedProducts(line.entries!!.removed.size).localised,
                                    true
                                ) { entry ->
                                    vmBasketPreview.setEvent(
                                        BasketPreviewContract.Event.AddEntry(entry)
                                    )
                                }
                            }
                        }
                    }
                }
            }
        )
    }

    @Composable
    private fun ProductSection(entriesList: List<BasketEntry>, sectionTitle: String, canBeAdd: Boolean, add: (entrie: BasketEntry) -> Unit) {

        var expandedState by remember { mutableStateOf(false) }

        Column() {
            TemplateDI.mealPlannerBasketPreviewSectionTitle.view.Content(
                mealPlannerBasketPreviewSectionTitleParameters = MealPlannerBasketPreviewSectionTitleParameters(
                    sectionTitle,
                    expandedState
                ) {
                    expandedState = !expandedState
                }
            )
            AnimatedVisibility(visible = expandedState) {
                Column() {
                    entriesList.forEach { basketEntry ->
                        TemplateDI.mealPlannerBasketPreviewSectionProduct.view.Content(
                            mealPlannerBasketPreviewSectionProductParameters = MealPlannerBasketPreviewSectionProductParameters(
                                basketEntry.relationships?.groceriesEntry?.data?.attributes?.name ?: "",
                                canBeAdd
                            ) {
                                add(basketEntry)
                            }
                        )
                    }
                }
            }
        }
    }
}