package com.miam.sdk.ui.mealPlanner.recipe

import androidx.compose.foundation.BorderStroke
import androidx.compose.foundation.Canvas
import androidx.compose.foundation.Image
import androidx.compose.foundation.background
import androidx.compose.foundation.border
import androidx.compose.foundation.clickable
import androidx.compose.foundation.gestures.detectHorizontalDragGestures
import androidx.compose.foundation.layout.*
import androidx.compose.foundation.shape.RoundedCornerShape
import androidx.compose.material.Button
import androidx.compose.material.ButtonDefaults
import androidx.compose.material.Card
import androidx.compose.material.Text
import androidx.compose.material.TextButton
import androidx.compose.runtime.Composable
import androidx.compose.runtime.getValue
import androidx.compose.runtime.mutableStateOf
import androidx.compose.runtime.remember
import androidx.compose.runtime.setValue
import androidx.compose.ui.Alignment
import androidx.compose.ui.Modifier
import androidx.compose.ui.geometry.Offset
import androidx.compose.ui.graphics.Color
import androidx.compose.ui.graphics.ColorFilter
import androidx.compose.ui.graphics.painter.Painter
import androidx.compose.ui.input.pointer.pointerInput
import androidx.compose.ui.layout.ContentScale
import androidx.compose.ui.res.painterResource
import androidx.compose.ui.text.font.FontWeight
import androidx.compose.ui.text.style.TextAlign
import androidx.compose.ui.text.style.TextOverflow
import androidx.compose.ui.tooling.preview.Preview
import androidx.compose.ui.unit.Dp
import androidx.compose.ui.unit.IntOffset
import androidx.compose.ui.unit.dp
import coil.compose.rememberImagePainter
import com.miam.core.sdk.localisation.Localisation
import com.miam.kmmMiamCore.miam_core.model.RecipeAttributes
import com.miam.kmmMiamCore.miam_core.model.fake.RecipeFakeFactory
import com.miam.kmm_miam_sdk.android.ressource.Image.delete
import com.miam.kmm_miam_sdk.android.ressource.Image.miamDifficulty
import com.miam.kmm_miam_sdk.android.ressource.Image.miamTime
import com.miam.kmm_miam_sdk.android.ressource.Image.plus
import com.miam.kmm_miam_sdk.android.theme.Colors
import com.miam.kmm_miam_sdk.android.theme.Colors.primary
import com.miam.kmm_miam_sdk.android.theme.Dimension
import com.miam.kmm_miam_sdk.android.theme.Typography
import com.miam.kmm_miam_sdk.android.ui.components.likeButton.LikeButton
import com.miam.sdk.templateInterfaces.mealPlanner.recipe.MealPlannerRecipeCard
import com.miam.sdk.templateParameters.mealPlanner.recipe.MealPlannerRecipeCardParameters
import kotlin.math.roundToInt

class MealPlannerRecipeCardImp: MealPlannerRecipeCard {
    @Composable
    override fun Content(mealPlannerRecipeCardParameters: MealPlannerRecipeCardParameters) {
        if (mealPlannerRecipeCardParameters.isInSearchPage) {
            return RecipeCardSearchPageContent(params = mealPlannerRecipeCardParameters)
        }
        RecipeCardContent(params = mealPlannerRecipeCardParameters)
    }

    @Preview
    @Composable
    fun MealPlannerRecipeCardImpPreview() {
        MealPlannerRecipeCardImp().Content(mealPlannerRecipeCardParameters = MealPlannerRecipeCardParameters(
            index = 1,
            isInSearchPage = false,
            isAlreadyInBasket = false,
            price = 45.0,
            recipe = RecipeFakeFactory.create(attributes = RecipeAttributes(title = "Deep Dish", mediaUrl = "https://picsum.photos/200/300")),
            changeAction = {},
            replaceAction = { },
            deleteAction = {},
            openDetail = {}
        ))
    }
}

@Composable
fun RecipeCardContent(params: MealPlannerRecipeCardParameters) {
    var offsetX by remember { mutableStateOf(0f) }
    val threshold = 200f
    val maxSwipeDistance = 250f // Change this value to adjust the maximum swipe distance
    val showCloseIcon = remember { mutableStateOf(false) }

    Box(Modifier.padding(vertical = 8.dp)) {
        Button(
            onClick = {
                params.deleteAction()
            },
            colors = ButtonDefaults.buttonColors(
                backgroundColor = Colors.miamDangerBackground
            ),
            modifier = Modifier
                .align(Alignment.CenterEnd)
                .height(Dimension.mealPlannerCardHeight)
                .width(maxSwipeDistance.dp)
        ) {
            Row(
                modifier = Modifier.fillMaxSize(),
                horizontalArrangement = Arrangement.End,
                verticalAlignment = Alignment.CenterVertically
            ) {
                Image(
                    painter = painterResource(delete),
                    contentDescription = null,
                    colorFilter = ColorFilter.tint(Colors.miamDangerText),
                    modifier = Modifier.size(30.dp)
                )
            }
        }
        Box(
            modifier = Modifier
                .height(Dimension.mealPlannerCardHeight)
                .offset { IntOffset(offsetX.roundToInt(), 0) }
                .pointerInput(Unit) {
                    detectHorizontalDragGestures { change, dragAmount ->
                        val newOffsetX = offsetX + dragAmount
                        if (newOffsetX <= 0f && newOffsetX >= -maxSwipeDistance) {
                            offsetX = newOffsetX
                        }
                        change.consume()
                        showCloseIcon.value = offsetX <= -threshold
                    }
                }
        ) {
            RecipeCardFace(params)
        }
    }
}

@Composable
fun RecipeCardFace(params: MealPlannerRecipeCardParameters) {
    Box(
        modifier = Modifier
            .height(Dimension.mealPlannerCardHeight)
            .background(color = Color.White)
    ) {
        Canvas(modifier = Modifier.matchParentSize()) {
            val strokeWidthPx = Dimension.borderWidth.toPx()
            val yTop = strokeWidthPx / 4
            val yBottom = size.height - strokeWidthPx / 4

            drawLine(
                color = Color.LightGray,
                Offset(x = 0f, y = yTop),
                Offset(size.width, yTop),
                strokeWidthPx
            )
            drawLine(
                color = Color.LightGray,
                Offset(x = 0f, y = yBottom),
                Offset(size.width, yBottom),
                strokeWidthPx
            )
        }
        Row(
            verticalAlignment = Alignment.CenterVertically,
        ) {
            Box {
                val backgroundImage: Painter = rememberImagePainter(params.recipe.attributes!!.mediaUrl)
                Image(
                    painter = backgroundImage,
                    contentDescription = "Grocery Item Image",
                    modifier = Modifier
                        .height(Dimension.mealPlannerCardHeight)
                        .width(150.dp)
                        .clickable { params.openDetail() },
                    contentScale = ContentScale.Crop
                )
                BudgetLikeButton(params.recipe.id, 150.dp)
            }
            Column(
                modifier = Modifier
                    .fillMaxSize()
                    .padding(Dimension.lPadding),
                verticalArrangement = Arrangement.SpaceBetween,
                horizontalAlignment = Alignment.CenterHorizontally,
            ) {
                Text(
                    text = params.recipe.attributes!!.title,
                    style = Typography.link,
                    textAlign = TextAlign.Center,
                    maxLines = 2,
                    overflow = TextOverflow.Ellipsis,
                    modifier = Modifier.padding(horizontal = Dimension.lPadding)
                )
                Row(
                    Modifier.fillMaxWidth(),
                    verticalAlignment = Alignment.CenterVertically,
                    horizontalArrangement = Arrangement.Center
                ) {
                    when (params.recipe.attributes!!.difficulty) {
                        1 -> BudgetMetricView(Localisation.Recipe.lowDifficulty.localised, miamDifficulty)
                        2 -> BudgetMetricView(Localisation.Recipe.mediumDifficulty.localised, miamDifficulty)
                        3 -> BudgetMetricView(Localisation.Recipe.highDifficulty.localised, miamDifficulty)
                    }
                    BudgetMetricView(params.recipe.totalTime, miamTime)

                }
                TextButton(
                    onClick = { params.changeAction() },
                    colors = ButtonDefaults.textButtonColors(contentColor = Colors.primary)
                ) {
                    Text(text = Localisation.Budget.replaceItem.localised, style = Typography.link)
                }
            }
        }
    }
}

@Composable
fun RecipeCardSearchPageContent(params: MealPlannerRecipeCardParameters) {

    Card(
        Modifier.fillMaxWidth()
    ) {
        Column(horizontalAlignment = Alignment.CenterHorizontally) {
            Box {
                val backgroundImage: Painter = rememberImagePainter(params.recipe.attributes!!.mediaUrl)
                Image(
                    painter = backgroundImage,
                    contentDescription = "Grocery Item Image",
                    modifier = Modifier
                        .height(Dimension.mealPlannerCardHeight)
                        .fillMaxWidth()
                        .clickable { params.openDetail() },
                    contentScale = ContentScale.Crop
                )
                BudgetLikeButton(params.recipe.id)
            }
            Spacer(modifier = Modifier.height(12.dp))
            Text(
                text = params.recipe.attributes!!.title,
                style = Typography.link,
                textAlign = TextAlign.Center,
                maxLines = 2,
                overflow = TextOverflow.Ellipsis,
                modifier = Modifier
                    .padding(horizontal = Dimension.lPadding)
                    .height(40.dp)
            )
            Spacer(modifier = Modifier.height(12.dp))
            Row(
                Modifier.fillMaxWidth(),
                verticalAlignment = Alignment.CenterVertically,
                horizontalArrangement = Arrangement.Center
            ) {

                BudgetMetricView(params.recipe.totalTime, miamTime)
                Box(Modifier.weight(1f)) {
                    when (params.recipe.attributes!!.difficulty) {
                        1 -> BudgetMetricView(Localisation.Recipe.lowDifficulty.localised, miamDifficulty)
                        2 -> BudgetMetricView(Localisation.Recipe.mediumDifficulty.localised, miamDifficulty)
                        3 -> BudgetMetricView(Localisation.Recipe.highDifficulty.localised, miamDifficulty)
                    }
                }
            }
            Spacer(modifier = Modifier.height(12.dp))
            Box(
                Modifier
                    .fillMaxWidth()
                    .padding(horizontal = 8.dp)
                    .background(color = Colors.white, shape = RoundedCornerShape(8.dp))
                    .border(BorderStroke(1.dp, Colors.primary), shape = RoundedCornerShape(8.dp))
                    .clickable {
                        params.replaceAction()
                    }
            ) {
                Row(
                    modifier = Modifier
                        .padding(horizontal = 16.dp, vertical = 8.dp)
                        .height(50.dp)
                        .fillMaxWidth(),
                    verticalAlignment = Alignment.CenterVertically,
                    horizontalArrangement = Arrangement.Center
                ) {
                    Image(
                        painter = painterResource(plus),
                        contentDescription = "less icon",
                        colorFilter = ColorFilter.tint(primary),
                        modifier = Modifier.size(16.dp)
                    )
                    Spacer(modifier = Modifier.width(12.dp))
                    Text(
                        text = "Ajouter", // TODO ADD TRANSLATE
                        color = primary,
                        style = Typography.bodyBold,
                    )
                }


            }
            Spacer(modifier = Modifier.height(12.dp))

        }


    }


}

@Composable
fun BudgetLikeButton(recipeId: String, sizeOfImage: Dp? = null) {
    Row(
        modifier = Modifier
            .then(
                if (sizeOfImage != null) Modifier.width(sizeOfImage)
                else Modifier.fillMaxWidth()
            )
            .padding(top = Dimension.mPadding, end = Dimension.mPadding),
        horizontalArrangement = Arrangement.End
    ) {
        val likeButton = LikeButton(backgroundShape = RoundedCornerShape(Dimension.lRoundedCorner))
        likeButton.bind(recipeId)
        likeButton.Content()
    }
}

@Composable
fun BudgetMetricView(text: String, image: Int) {
    Column(
        horizontalAlignment = Alignment.CenterHorizontally,
        verticalArrangement = Arrangement.Top,
        modifier = Modifier
            .padding(horizontal = 16.dp)
    ) {
        Image(
            painter = painterResource(image),
            contentDescription = null,
            colorFilter = ColorFilter.tint(Colors.boldText),
            modifier = Modifier.size(22.dp)
        )
        Spacer(modifier = Modifier.height(4.dp))
        Text(
            text = text,
            color = Colors.boldText,
            style = Typography.overLine.copy(fontWeight = FontWeight.Bold),
            textAlign = TextAlign.Center,
            modifier = Modifier.padding(top = 8.dp)
        )
    }
}