package com.miam.sdk.ui.mealPlanner.basketPreview

import androidx.compose.animation.core.animateFloatAsState
import androidx.compose.foundation.BorderStroke
import androidx.compose.foundation.Image
import androidx.compose.foundation.background
import androidx.compose.foundation.border
import androidx.compose.foundation.clickable
import androidx.compose.foundation.layout.*
import androidx.compose.foundation.shape.RoundedCornerShape
import androidx.compose.material.CircularProgressIndicator
import androidx.compose.material.Divider
import androidx.compose.material.IconButton
import androidx.compose.material.Text
import androidx.compose.runtime.Composable
import androidx.compose.runtime.getValue
import androidx.compose.runtime.mutableStateOf
import androidx.compose.runtime.remember
import androidx.compose.runtime.setValue
import androidx.compose.ui.Alignment
import androidx.compose.ui.Modifier
import androidx.compose.ui.draw.clip
import androidx.compose.ui.draw.rotate
import androidx.compose.ui.graphics.ColorFilter
import androidx.compose.ui.layout.ContentScale
import androidx.compose.ui.res.painterResource
import androidx.compose.ui.text.style.TextOverflow
import androidx.compose.ui.unit.dp
import coil.compose.rememberImagePainter
import com.miam.core.sdk.localisation.Localisation
import com.miam.kmm_miam_sdk.android.ressource.Image
import com.miam.kmm_miam_sdk.android.theme.Colors
import com.miam.kmm_miam_sdk.android.theme.Colors.primary
import com.miam.kmm_miam_sdk.android.theme.Typography
import com.miam.kmm_miam_sdk.android.ui.components.common.Clickable
import com.miam.kmm_miam_sdk.android.ui.components.counter.CounterColor
import com.miam.kmm_miam_sdk.android.ui.components.price.formatPrice
import com.miam.sdk.templateInterfaces.mealPlanner.basketPreview.MealPlannerBasketPreviewRecipeOverview
import com.miam.sdk.templateParameters.mealPlanner.basketPreview.MealPlannerBasketPreviewRecipeOverviewParameters

class MealPlannerBasketPreviewRecipeOverviewImp: MealPlannerBasketPreviewRecipeOverview {

    private fun newValueBounded(newValue: Int): Boolean {
        return (newValue >= 1) && (newValue <= 100)
    }

    private fun changedValue(localCount: Int?, delta: Int): Int? {

        if (localCount == null) return 1

        if (!newValueBounded(localCount + delta)) return null

        return localCount + delta
    }

    @Composable
    override fun Content(mealPlannerBasketPreviewRecipeOverviewParameters: MealPlannerBasketPreviewRecipeOverviewParameters) {

        Column {
            Row(
                verticalAlignment = Alignment.Top,
                modifier = Modifier
                    .fillMaxWidth()
                    .padding(12.dp)
            ) {
                Clickable(
                    onClick = {
                        if (mealPlannerBasketPreviewRecipeOverviewParameters.isDeleting) return@Clickable
                        mealPlannerBasketPreviewRecipeOverviewParameters.openRecipeDetail()
                    },
                ) {
                    Image(
                        painter = rememberImagePainter(mealPlannerBasketPreviewRecipeOverviewParameters.picture),
                        contentDescription = null,
                        contentScale = ContentScale.Crop,
                        modifier = Modifier
                            .height(120.dp)
                            .width(120.dp)
                            .clip(RoundedCornerShape(8.dp)),
                    )
                }
                Spacer(modifier = Modifier.padding(horizontal = 16.dp))
                Column(
                    verticalArrangement = Arrangement.Top
                ) {
                    Row(
                        Modifier.fillMaxWidth(),
                        verticalAlignment = Alignment.Top,
                        horizontalArrangement = Arrangement.SpaceBetween
                    ) {
                        Clickable(
                            onClick = {
                                if (mealPlannerBasketPreviewRecipeOverviewParameters.isDeleting) return@Clickable
                                mealPlannerBasketPreviewRecipeOverviewParameters.openRecipeDetail()
                            },
                            Modifier.weight(1f)
                        ) {
                            Text(
                                text = mealPlannerBasketPreviewRecipeOverviewParameters.name,
                                maxLines = 2,
                                color = Colors.black,
                                overflow = TextOverflow.Ellipsis,
                                style = Typography.bodyBold,
                            )
                        }
                        Column() {
                            Spacer(modifier = Modifier.height(8.dp))
                            if (mealPlannerBasketPreviewRecipeOverviewParameters.isDeleting) {
                                CircularProgressIndicator(
                                    Modifier
                                        .size(20.dp),
                                    primary,
                                    1.dp
                                )
                            } else {
                                IconButton(
                                    modifier = Modifier
                                        .size(20.dp),
                                    onClick = {
                                        mealPlannerBasketPreviewRecipeOverviewParameters.delete()
                                    }
                                ) {
                                    Image(
                                        painter = painterResource(Image.delete),
                                        contentDescription = "delete",
                                        colorFilter = ColorFilter.tint(primary),
                                    )
                                }
                            }
                        }
                    }
                    Spacer(modifier = Modifier.padding(vertical = 4.dp))
                    Text(
                        text = "${mealPlannerBasketPreviewRecipeOverviewParameters.productCount} articles",
                        color = Colors.grey,
                        style = Typography.bodySmall
                    )

                    Spacer(modifier = Modifier.padding(vertical = 4.dp))
                    Row() {
                        ToggleOverlinedButton(
                            "Recommandé", //TODO put translate
                            mealPlannerBasketPreviewRecipeOverviewParameters.isRecommendedProduct
                        )
                        {
                            mealPlannerBasketPreviewRecipeOverviewParameters.toggleProductRecommendation
                        }
                        Spacer(modifier = Modifier.padding(horizontal = 4.dp))
                        ToggleOverlinedButton(
                            "Petit prix", //TODO put translate
                            !mealPlannerBasketPreviewRecipeOverviewParameters.isRecommendedProduct
                        )
                        {
                            mealPlannerBasketPreviewRecipeOverviewParameters.toggleProductRecommendation
                        }
                    }
                }
            }
            Row(
                Modifier
                    .fillMaxWidth()
                    .padding(16.dp),
                horizontalArrangement = Arrangement.SpaceBetween,
                verticalAlignment = Alignment.CenterVertically
            ) {
                PriceInfo(
                    mealPlannerBasketPreviewRecipeOverviewParameters.totalPrice,
                    mealPlannerBasketPreviewRecipeOverviewParameters.guestNumber
                )
                Row(verticalAlignment = Alignment.CenterVertically) {
                    Counter(
                        initialCount = mealPlannerBasketPreviewRecipeOverviewParameters.guestNumber,
                        isLoading = mealPlannerBasketPreviewRecipeOverviewParameters.isDeleting 
                    ) {
                        mealPlannerBasketPreviewRecipeOverviewParameters.onGuestChange(it)
                    }
                    Spacer(modifier = Modifier.padding(horizontal = 8.dp))
                    CollapseButton(mealPlannerBasketPreviewRecipeOverviewParameters.isCollapsed) {
                        mealPlannerBasketPreviewRecipeOverviewParameters.toggleCollapse()
                    }
                }
            }
            Divider()
            mealPlannerBasketPreviewRecipeOverviewParameters.Content()
        }
    }

    @Composable
    private fun ToggleOverlinedButton(text: String, isToggle: Boolean, action: () -> Unit) {
        Box(
            Modifier
                .background(color = if (isToggle) Colors.primary else Colors.white, shape = RoundedCornerShape(8.dp))
                .border(BorderStroke(1.dp, Colors.primary), shape = RoundedCornerShape(8.dp))
                .clickable {
                    action()
                }
        ) {
            Text(
                text = text,
                modifier = Modifier.padding(8.dp),
                color = if (isToggle) Colors.white else primary,
                style = Typography.body
            )
        }
    }

    @Composable
    fun PriceInfo(price: Double, guestCount: Int) {
        Column(modifier = Modifier.padding(bottom = 4.dp)) {
            Column(
                horizontalAlignment = Alignment.CenterHorizontally
            ) {
                Row() {
                    Text(
                        formatPrice(price),
                        color = primary,
                        style = Typography.subtitleBold
                    )
                }
            }
            Text(
                text = formatPrice(price / guestCount) + Localisation.Price.perGuest.localised,
                color = Colors.grey,
                style = Typography.overLine
            )
        }
    }

    @Composable
    private fun CollapseButton(initaleState: Boolean, action: () -> Unit) {

        var expandedState by remember { mutableStateOf(initaleState) }
        val rotationState by animateFloatAsState(
            targetValue = if (expandedState) 90f else 0f
        )

        IconButton(
            modifier = Modifier
                .size(30.dp)
                .rotate(rotationState),
            onClick = {
                expandedState = !expandedState
                action()
            }
        ) {
            Image(
                painter = painterResource(Image.toggleCaret),
                contentDescription = null,
                modifier = Modifier.size(24.dp),
                colorFilter = ColorFilter.tint(primary),
            )
        }
    }


    @Composable
    private fun Counter(initialCount: Int, isLoading: Boolean, onCounterChanged: (Int) -> Unit) {

        var localCount by remember { mutableStateOf(initialCount) }

        fun increase() {
            changedValue(localCount, 1)?.let { newCount ->
                localCount = newCount
                onCounterChanged(newCount)
            }
        }

        fun decrease() {
            changedValue(localCount, -1)?.let { newCount ->
                localCount = newCount
                onCounterChanged(newCount)
            }
        }

        Row(
            verticalAlignment = Alignment.CenterVertically,
            modifier = Modifier.border(BorderStroke(1.dp, primary), shape = RoundedCornerShape(8.dp))
        ) {

            CounterButton(Image.less, isLoading) { decrease() }
            CounterText(localCount, isLoading)
            CounterButton(Image.plus, isLoading) { increase() }
        }
    }

    @Composable
    private fun CounterButton(image: Int, isDisable: Boolean, action: () -> Unit) {

        IconButton(
            onClick = { action() },
            enabled = !isDisable,
            modifier = Modifier
                .clip(RoundedCornerShape(8.dp))
                .background(if (isDisable) Colors.lightgrey else Colors.white)
        ) {
            Image(
                painter = painterResource(image),
                contentDescription = "less icon",
                colorFilter = ColorFilter.tint(primary),
                modifier = Modifier.size(14.dp)
            )
        }
    }

    @Composable
    private fun CounterText(localCount: Int?, isLoading: Boolean) {

        Row(
            modifier = Modifier,
            verticalAlignment = Alignment.CenterVertically,
            horizontalArrangement = Arrangement.Center,
        ) {
            if (isLoading) {
                CircularProgressIndicator(color = CounterColor.countTextColor)
            } else {
                Text(
                    text = localCount.toString() + " " + Localisation.Counter.persons.localised,
                    color = Colors.black,
                    style = Typography.bodyBold
                )
            }
        }
    }
}