package com.miam.sdk.ui.budget.form.expandedForm

import androidx.compose.foundation.background
import androidx.compose.foundation.border
import androidx.compose.foundation.layout.*
import androidx.compose.foundation.shape.RoundedCornerShape
import androidx.compose.foundation.text.BasicTextField
import androidx.compose.foundation.text.KeyboardOptions
import androidx.compose.material.Divider
import androidx.compose.material.LocalTextStyle
import androidx.compose.material.Text
import androidx.compose.runtime.Composable
import androidx.compose.runtime.getValue
import androidx.compose.runtime.mutableStateOf
import androidx.compose.runtime.remember
import androidx.compose.runtime.setValue
import androidx.compose.ui.Alignment
import androidx.compose.ui.Modifier
import androidx.compose.ui.graphics.Color
import androidx.compose.ui.text.TextStyle
import androidx.compose.ui.text.font.FontWeight
import androidx.compose.ui.text.input.KeyboardType
import androidx.compose.ui.text.style.TextAlign
import androidx.compose.ui.tooling.preview.Preview
import androidx.compose.ui.unit.dp
import androidx.compose.ui.unit.sp
import com.miam.core.sdk.localisation.Localisation
import com.miam.kmm_miam_sdk.android.theme.Colors.black
import com.miam.kmm_miam_sdk.android.theme.Colors.white

@Composable
fun MealPlannerBudget(
    modifier: Modifier,
    caption: String? = null,
    currency: String,
    budget: Int,
    onBudgetChange: (newBudget: Int) -> Unit,
) {

    var currentBudget by remember { mutableStateOf(budget) }
    val borderColor = Color(0XFFD9DDE1)

    Column {
        caption?.let {
            Text(
                text = it,
                style = TextStyle(color = black, fontSize = 14.sp, fontWeight = FontWeight.W700)
            )

            Spacer(modifier = Modifier.height(4.dp))
        }
        Row(
            modifier = modifier
                .height(48.dp)
                .border(
                    width = 1.dp,
                    color = borderColor,
                    shape = RoundedCornerShape(6.dp)
                ),
            verticalAlignment = Alignment.CenterVertically,
        ) {
            Currency(currency)
            Divider(modifier = Modifier.size(width = 1.dp, height = 48.dp), color = borderColor)
            Budget(currentBudget) { newText ->
                currentBudget = newText
                onBudgetChange(currentBudget)
            }
        }
    }
}

@Composable
fun Currency(text: String) {
    Box(contentAlignment = Alignment.Center) {
        Text(
            modifier = Modifier
                .width(48.dp)
                .wrapContentHeight(),
            text = text,
            textAlign = TextAlign.Center,
            style = TextStyle(color = black, fontSize = 14.sp, fontWeight = FontWeight.W500)
        )
    }
}

@Composable
fun Budget(budgetAmount: Int, onBudgetChanged: (Int) -> Unit) {
    BasicTextField(
        modifier = Modifier
            .defaultMinSize(minHeight = 48.dp)
            .fillMaxWidth()
            .wrapContentHeight(Alignment.CenterVertically)
            .padding(start = 8.dp),
        value = budgetAmount.toString(),
        keyboardOptions = KeyboardOptions(keyboardType = KeyboardType.Number),
        onValueChange = { newValue ->
            onBudgetChanged(newValue.toIntOrNull() ?: 0)
        },
        singleLine = true,
        textStyle = LocalTextStyle.current.copy(textAlign = TextAlign.Start)
    )
}

@Preview
@Composable
fun MealPlannerBudgetPreview() {
    Box(modifier = Modifier.background(white)) {
        MealPlannerBudget(
            modifier = Modifier.fillMaxWidth(),
            caption = Localisation.Budget.totalBudgetTitle.localised,
            currency = Localisation.Price.currency.localised,
            budget = 3,
            onBudgetChange = {})
    }
}