package com.miam.kmm_miam_sdk.android.ui.components.price

import androidx.compose.animation.AnimatedVisibility
import androidx.compose.animation.core.FastOutLinearInEasing
import androidx.compose.animation.core.animateFloat
import androidx.compose.animation.core.infiniteRepeatable
import androidx.compose.animation.core.rememberInfiniteTransition
import androidx.compose.animation.core.tween
import androidx.compose.animation.slideInVertically
import androidx.compose.animation.slideOutVertically
import androidx.compose.foundation.Image
import androidx.compose.foundation.background
import androidx.compose.foundation.layout.*
import androidx.compose.foundation.shape.RoundedCornerShape
import androidx.compose.material.Text
import androidx.compose.runtime.*
import androidx.compose.ui.Alignment
import androidx.compose.ui.Modifier
import androidx.compose.ui.geometry.Offset
import androidx.compose.ui.graphics.Brush
import androidx.compose.ui.res.painterResource
import androidx.compose.ui.text.style.TextAlign
import androidx.compose.ui.tooling.preview.Preview
import androidx.compose.ui.unit.dp
import com.miam.core.sdk.localisation.Localisation
import com.miam.kmmMiamCore.component.pricing.RecipePricingViewModel
import com.miam.kmmMiamCore.miam_core.model.Pricing
import com.miam.kmm_miam_sdk.android.ressource.Image
import com.miam.kmm_miam_sdk.android.templatesParameters.PriceParameters
import com.miam.kmm_miam_sdk.android.theme.Colors
import com.miam.kmm_miam_sdk.android.theme.Template
import com.miam.kmm_miam_sdk.android.theme.Typography
import com.miam.kmm_miam_sdk.android.theme.Typography.body
import com.miam.kmm_miam_sdk.android.theme.Typography.bodySmall
import com.miam.kmm_miam_sdk.android.theme.Typography.subtitleBold
import com.miam.kmm_miam_sdk.android.ui.components.price.PriceColor.loaderColor
import com.miam.kmm_miam_sdk.android.ui.components.price.PriceColor.priceIntegerColor
import com.miam.kmm_miam_sdk.android.ui.components.price.PriceColor.priceIntegerTotalPriceColor
import com.miam.kmm_miam_sdk.android.ui.components.price.PriceColor.subtitleColor
import com.miam.kmm_miam_sdk.android.ui.components.price.PriceStyle.loaderInteger
import com.miam.kmm_miam_sdk.android.ui.components.price.PriceStyle.mainContainer
import com.miam.kmm_miam_sdk.android.ui.components.price.PriceStyle.priceContainer
import com.miam.kmm_miam_sdk.android.ui.components.price.PriceStyle.priceEmptyState
import com.miam.kmm_miam_sdk.android.ui.components.price.PriceText.preGuests
import com.miam.kmm_miam_sdk.android.ui.components.states.ManagementResourceState
import java.text.NumberFormat
import java.util.*

fun formatPrice(price: Double): String {
    val numberFormat = NumberFormat.getCurrencyInstance()
    numberFormat.currency = Currency.getInstance(Localisation.Price.currency.localised)
    return numberFormat.format(price)
}


@Composable
fun SimplePrice(price: Double, successView: (@Composable (price: Double) -> Unit)? = null) {
    if (Template.simplePriceTemplate != null) {
        Template.simplePriceTemplate?.let { it(PriceParameters(price)) }
    } else {
        successView?.let {
            successView(price)
        } ?: run {
            Column(
                modifier = mainContainer,
                horizontalAlignment = Alignment.CenterHorizontally
            ) {
                Row(modifier = priceContainer) {
                    Text(
                        formatPrice(price),
                        color = priceIntegerTotalPriceColor,
                        style = subtitleBold
                    )
                }
            }
        }
    }
}

@Composable
fun RecipePrice(
    recipeId: String,
    guestNumber: Int,
    isFooter: Boolean = false,
    successView: (@Composable (price: Double) -> Unit)? = null,
    loadingView: (@Composable () -> Unit) = { PriceShimmer() }
) {
    val vmPrice = RecipePricingViewModel()
    vmPrice.setRecipe(recipeId, guestNumber)
    PriceStateManager(vmPrice, isFooter, successView, loadingView)

    LaunchedEffect(Unit) { vmPrice.listenBasketChanges() }
    DisposableEffect(Unit) { onDispose { vmPrice.stopListenBasketChanges() } }
}

@Composable
fun PriceStateManager(
    vmPrice: RecipePricingViewModel,
    isFooter: Boolean,
    successView: (@Composable (price: Double) -> Unit)? = null,
    loadingView: (@Composable () -> Unit) = { PriceShimmer() }
) {
    val state by vmPrice.uiState.collectAsState()
    var isLoaded by remember { mutableStateOf(false) }
    Box {
        ManagementResourceState(
            resourceState = state.price,
            successView = { price ->
                requireNotNull(price)
                LaunchedEffect(price) { isLoaded = true }
                if (isFooter)
                    AnimatedVisibility(
                        visible = isLoaded,
                        enter = slideInVertically { height -> height },
                        exit = slideOutVertically { height -> height }
                    ) {
                        recipeDetailPriceFooter(price)
                    }
                else successView?.let { it(price.price) } ?: run {
                    RecipePriceView(price.pricePerServe)
                }

            },
            emptyView = { EmptyState() },
            loadingView = { if (isFooter) Box {} else loadingView() }
        )
    }
}

@Composable
fun EmptyState() {
    if (Template.priceEmptyTemplate != null) {
        Template.priceEmptyTemplate?.let { it() }
    } else {
        Spacer(modifier = priceEmptyState)
    }
}

@Composable
fun RecipePriceView(price: Double) {

    if (Template.recipePriceTemplate != null) {
        Template.recipePriceTemplate?.let { it(PriceParameters(price)) }
    } else {
        Column(
            modifier = mainContainer,
            horizontalAlignment = Alignment.CenterHorizontally
        ) {
            Row(modifier = priceContainer) {
                Text(
                    formatPrice(price),
                    color = priceIntegerColor,
                    style = subtitleBold
                )
            }
            Text(
                preGuests,
                color = subtitleColor,
                style = bodySmall
            )
        }
    }
}

@Composable
fun recipeDetailPriceFooter(price: Pricing) {
    Row(
        Modifier
            .fillMaxWidth()
            .padding(16.dp),
        horizontalArrangement = Arrangement.SpaceBetween,
        verticalAlignment = Alignment.CenterVertically
    ) {
        Row() {
            Text(
                formatPrice(price.pricePerServe),
                color = priceIntegerColor,
                style = subtitleBold
            )
            Text(
                preGuests,
                style = body,
                modifier = Modifier.padding(start = 8.dp)
            )
        }
        Box(Modifier.background(Colors.miamSuccessBackGround, shape = RoundedCornerShape(8.dp))) {
            Row(Modifier.padding(horizontal = 20.dp, vertical = 8.dp)) {
                Text(text = "Soit ")
                Box() {
                    Text(
                        text = formatPrice(price.price),
                        style = Typography.bodyBold,
                        textAlign = TextAlign.Center

                    )
                    Image(
                        modifier = Modifier
                            .offset(0.dp, 20.dp)
                            .width(50.dp),
                        painter = painterResource(id = Image.trait),
                        contentDescription = ""
                    )
                }
                Text(text = " le repas")
            }
        }
    }
}


@Composable
fun PriceShimmer() {
    if (Template.priceLoadingTemplate != null) {
        Template.priceLoadingTemplate?.let { it() }
    } else {
        val shimerColors = listOf(
            loaderColor.copy(alpha = 0.6F),
            loaderColor.copy(alpha = 0.2F),
            loaderColor.copy(alpha = 0.6F)
        )

        val transition = rememberInfiniteTransition()
        val translateAnimation = transition.animateFloat(
            initialValue = 0f,
            targetValue = 1000f,
            animationSpec = infiniteRepeatable(
                animation = tween(
                    durationMillis = 1000,
                    easing = FastOutLinearInEasing
                )
            )
        )

        val brush = Brush.linearGradient(
            colors = shimerColors,
            start = Offset.Zero,
            end = Offset(
                x = translateAnimation.value,
                y = translateAnimation.value
            )
        )
        shimmerPriceItem(brush)
    }
}

@Composable
fun shimmerPriceItem(brush: Brush) {

    Column(
        horizontalAlignment = Alignment.CenterHorizontally
    ) {
        Row {
            Spacer(modifier = loaderInteger.background(brush = brush))
        }
        Text(
            preGuests,
            color = subtitleColor,
            style = bodySmall
        )
    }
}


@Preview
@Composable
fun PricePreview() {
    RecipePriceView(10.0)
}

@Preview
@Composable
fun PriceLoadingPreview() {
    PriceShimmer()
}



