/*
 * Decompiled with CFR 0.152.
 */
package tech.lastbox.jwt;

import com.auth0.jwt.JWT;
import com.auth0.jwt.algorithms.Algorithm;
import java.time.Instant;
import java.util.HashSet;
import java.util.List;
import java.util.Optional;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.transaction.annotation.Transactional;
import tech.lastbox.jwt.DateUtil;
import tech.lastbox.jwt.ExpirationTimeUnit;
import tech.lastbox.jwt.JwtConfig;
import tech.lastbox.jwt.Token;
import tech.lastbox.jwt.TokenCreationException;
import tech.lastbox.jwt.TokenEntity;
import tech.lastbox.jwt.TokenRevocationException;
import tech.lastbox.jwt.TokenStore;
import tech.lastbox.jwt.TokenUtil;
import tech.lastbox.jwt.TokenValidation;

public class JwtService {
    private final Algorithm algorithm;
    private final ExpirationTimeUnit expirationTimeUnit;
    private final HashSet<String> trustedIssuers;
    private final long expirationAmount;
    private final TokenStore tokenStore;
    private final Logger logger = LoggerFactory.getLogger(JwtService.class);

    public JwtService(JwtConfig jwtConfig) {
        this.algorithm = jwtConfig.getJwtAlgorithm().getAlgorithm(jwtConfig.getSecretKey());
        this.trustedIssuers = jwtConfig.getTrustedIssuers();
        this.expirationTimeUnit = jwtConfig.getExpirationTimeUnit();
        this.expirationAmount = jwtConfig.getExpirationAmount();
        this.tokenStore = jwtConfig.getTokenStore();
    }

    @Transactional
    public Token generateToken(String subject, String issuer, List<String> scope) {
        if (subject == null || subject.isEmpty()) {
            throw new TokenCreationException("Subject must not be null or empty.");
        }
        if (scope == null || scope.isEmpty()) {
            throw new TokenCreationException("Scope must not be null or empty.");
        }
        if (!this.trustedIssuers.contains(issuer)) {
            throw new TokenCreationException("Issuer must be in issuers trusted list.");
        }
        Instant now = Instant.now();
        Instant expiresIn = DateUtil.getExpirationDate(now, this.expirationAmount, this.expirationTimeUnit);
        String token = JWT.create().withSubject(subject).withExpiresAt(expiresIn).withIssuedAt(now).withIssuer(issuer).withClaim("scope", scope).sign(this.algorithm);
        if (this.tokenStore != null) {
            TokenEntity tokenEntity = new TokenEntity(token, now, expiresIn, subject, issuer, scope);
            this.tokenStore.save(tokenEntity);
        }
        this.logger.info("Generating token with subject: {}", (Object)subject);
        return new Token(token, subject, DateUtil.instantToLocalDateTime(now), DateUtil.instantToLocalDateTime(expiresIn), issuer, scope, false);
    }

    @Transactional
    public Token generateToken(String subject, String issuer) {
        if (subject == null || subject.isEmpty()) {
            throw new TokenCreationException("Subject must not be null or empty.");
        }
        if (!this.trustedIssuers.contains(issuer)) {
            throw new TokenCreationException("Issuer must be in issuers trusted list.");
        }
        Instant now = Instant.now();
        Instant expiresIn = DateUtil.getExpirationDate(now, this.expirationAmount, this.expirationTimeUnit);
        List<String> scope = List.of("name, username");
        String token = JWT.create().withSubject(subject).withExpiresAt(expiresIn).withIssuedAt(now).withIssuer(issuer).withClaim("scope", scope).sign(this.algorithm);
        if (this.tokenStore != null) {
            TokenEntity tokenEntity = new TokenEntity(token, now, expiresIn, subject, issuer, scope);
            this.tokenStore.save(tokenEntity);
        }
        this.logger.info("Generating token with subject: {}", (Object)subject);
        return new Token(token, subject, DateUtil.instantToLocalDateTime(now), DateUtil.instantToLocalDateTime(expiresIn), issuer, scope, false);
    }

    @Transactional
    public void revokeToken(String token) {
        if (this.tokenStore == null) {
            this.logger.error("Cannot revoke token without a configured store.");
            throw new TokenRevocationException("Token Store not configured.");
        }
        this.tokenStore.findById(token).ifPresentOrElse(tokenEntity -> {
            if (tokenEntity.isRevoked()) {
                this.logger.info("Token is already revoked: {}", (Object)token);
            } else {
                tokenEntity.setRevoked(true);
                this.tokenStore.save((TokenEntity)tokenEntity);
                this.logger.info("Token revoked successfully: {}", (Object)token);
            }
        }, () -> {
            this.logger.error("Cannot revoke non-existent token: {}", (Object)token);
            throw new TokenRevocationException("Token not found in the repository.");
        });
    }

    public TokenValidation validateToken(String token) {
        Optional<Token> tokenOptional = this.getToken(token);
        return new TokenValidation(tokenOptional, tokenOptional.isPresent());
    }

    public Optional<Token> getToken(String token) {
        if (token == null || token.isEmpty()) {
            this.logger.error("Token is null or empty");
            return Optional.empty();
        }
        if (this.tokenStore != null) {
            Optional<TokenEntity> tokenEntity = this.tokenStore.findById(token);
            if (tokenEntity.isEmpty() || !tokenEntity.get().isValid()) {
                return Optional.empty();
            }
            return tokenEntity.map(TokenUtil::convertEntityToToken);
        }
        return TokenUtil.validateDecodedToken(this.algorithm, this.trustedIssuers, token);
    }
}

