/*
 * Copyright 2024 LastBox
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package tech.lastbox.lastshield.basicauth.entity;

import io.swagger.v3.oas.annotations.media.Schema;
import jakarta.persistence.*;
import org.springframework.boot.autoconfigure.condition.ConditionalOnProperty;
import tech.lastbox.lastshield.security.core.annotations.Password;
import tech.lastbox.lastshield.security.core.annotations.Username;

import java.util.Objects;

/**
 * Represents a User entity in the system.
 * <p>
 * This class contains user information such as name, username, password, and role.
 */
@Entity
@Table(name = "users")
@ConditionalOnProperty(name = "lastshield.basicauth", havingValue = "true")
public class User {

    /**
     * The unique identifier for the user.
     * <p>
     * This field is automatically generated by the database.
     */
    @Id
    @GeneratedValue
    @Schema(description = "The unique identifier of the user.", example = "1")
    private Long id;

    /**
     * The full name of the user.
     * <p>
     * This field is a required field to store the user's name.
     */
    @Schema(description = "The full name of the user.", example = "John Doe")
    private String name;

    /**
     * The unique username for the user.
     * <p>
     * This field must be unique and is used for authentication purposes.
     */
    @Username
    @Column(unique = true)
    @Schema(description = "The unique username of the user.", example = "johndoe")
    private String username;

    /**
     * The password for the user account.
     * <p>
     * This field contains the encrypted password.
     */
    @Password
    @Schema(description = "The password for the user's account.", example = "password123")
    private String password;

    /**
     * The role assigned to the user.
     * <p>
     * By default, a user is assigned the "USER" role, but this can be modified as needed.
     */
    @Schema(description = "The role of the user.", example = "USER")
    private String role;

    /**
     * Default constructor for the User class.
     */
    public User() {}

    /**
     * Constructor for creating a new User with the specified name, username, and password.
     * <p>
     * The default role is set to "USER".
     *
     * @param name the name of the user.
     * @param username the unique username of the user.
     * @param password the password of the user.
     */
    public User(String name, String username, String password) {
        this.name = name;
        this.username = username;
        this.password = password;
        this.role = "USER";
    }

    public Long getId() {
        return id;
    }

    public String getName() {
        return name;
    }

    public void setName(String name) {
        this.name = name;
    }

    public String getUsername() {
        return username;
    }

    public void setUsername(String username) {
        this.username = username;
    }

    public String getPassword() {
        return password;
    }

    public void setPassword(String password) {
        this.password = password;
    }

    public String getRole() {
        return role;
    }

    public void setRole(String role) {
        this.role = role;
    }

    @Override
    public boolean equals(Object o) {
        if (!(o instanceof User user)) return false;
        return Objects.equals(username, user.username);
    }

    @Override
    public int hashCode() {
        return Objects.hash(username);
    }
}
