/*
 * Decompiled with CFR 0.152.
 */
package net.luminis.quic.cli;

import java.io.BufferedReader;
import java.io.File;
import java.io.IOException;
import java.io.InputStreamReader;
import java.net.InetSocketAddress;
import java.net.URI;
import java.net.URISyntaxException;
import java.net.http.HttpClient;
import java.net.http.HttpRequest;
import java.net.http.HttpResponse;
import java.nio.file.Files;
import java.nio.file.Path;
import java.time.Duration;
import java.time.Instant;
import java.util.Arrays;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.CompletionStage;
import java.util.function.Consumer;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import net.luminis.quic.QuicClientConnection;
import net.luminis.quic.cid.ConnectionIdInfo;
import net.luminis.quic.cid.ConnectionIdStatus;
import net.luminis.quic.cli.KwikCli;
import net.luminis.quic.core.QuicClientConnectionImpl;
import net.luminis.quic.core.TransportParameters;
import net.luminis.tls.util.ByteUtils;

public class InteractiveShell {
    private final Map<String, Consumer<String>> commands;
    private boolean running;
    private Map<String, String> history;
    private final QuicClientConnectionImpl.ExtendedBuilder builder;
    private QuicClientConnectionImpl quicConnection;
    private ClientParameters params;
    private KwikCli.HttpVersion httpVersion;
    private HttpClient httpClient;
    private CompletableFuture<HttpResponse<Path>> currentHttpGetResult;

    public InteractiveShell(QuicClientConnectionImpl.ExtendedBuilder builder, String alpn, KwikCli.HttpVersion httpVersion) {
        Objects.requireNonNull(builder);
        Objects.requireNonNull(alpn);
        this.builder = builder;
        builder.applicationProtocol(alpn);
        this.httpVersion = httpVersion;
        this.commands = new LinkedHashMap<String, Consumer<String>>();
        this.history = new LinkedHashMap<String, String>();
        this.setupCommands();
        this.params = new ClientParameters();
    }

    private void setupCommands() {
        this.commands.put("help", this::help);
        this.commands.put("set", this::setClientParameter);
        this.commands.put("scid_length", this::setScidLength);
        this.commands.put("connect", this::connect);
        this.commands.put("close", this::close);
        this.commands.put("get", this::httpGet);
        this.commands.put("stop", this::httpStop);
        this.commands.put("ping", this::sendPing);
        this.commands.put("params", this::printClientParams);
        this.commands.put("server_params", this::printServerParams);
        this.commands.put("cid_new", this::newConnectionIds);
        this.commands.put("cid_next", this::nextDestinationConnectionId);
        this.commands.put("cid_list", this::printConnectionIds);
        this.commands.put("cid_retire", this::retireConnectionId);
        this.commands.put("udp_rebind", this::changeUdpPort);
        this.commands.put("update_keys", this::updateKeys);
        this.commands.put("statistics", this::printStatistics);
        this.commands.put("!!", this::repeatLastCommand);
        this.commands.put("quit", this::quit);
    }

    private void repeatLastCommand(String arg) {
        if (this.history.size() > 0) {
            Map.Entry lastCommand = this.history.entrySet().stream().reduce((first, second) -> second).orElse(null);
            this.commands.get(lastCommand.getKey()).accept((String)lastCommand.getValue());
        }
    }

    public void start() {
        BufferedReader in = new BufferedReader(new InputStreamReader(System.in));
        try {
            System.out.println("\nThis is the KWIK interactive shell. Type a command or 'help'.");
            this.prompt();
            this.running = true;
            while (this.running) {
                String cmdLine = in.readLine();
                if (!cmdLine.isBlank()) {
                    String cmd = cmdLine.split(" ")[0];
                    List matchingCommands = this.commands.keySet().stream().filter(command -> command.startsWith(cmd)).collect(Collectors.toList());
                    if (matchingCommands.size() == 1) {
                        String matchingCommand = (String)matchingCommands.get(0);
                        Consumer<String> commandFunction = this.commands.get(matchingCommand);
                        try {
                            String commandArgs = cmdLine.substring(cmd.length()).trim();
                            commandFunction.accept(commandArgs);
                            if (!matchingCommand.startsWith("!")) {
                                this.history.put(matchingCommand, commandArgs);
                            }
                        }
                        catch (Exception error) {
                            this.error(error);
                        }
                    } else {
                        this.unknown(cmd);
                    }
                }
                if (!this.running) continue;
                this.prompt();
            }
        }
        catch (IOException e) {
            System.out.println("Error: " + e);
        }
    }

    private void connect(String arg) {
        int connectionTimeout = 3000;
        if (arg != null && !arg.isBlank()) {
            try {
                connectionTimeout = Integer.parseInt(arg);
                if (connectionTimeout < 100) {
                    System.out.println("Connection timeout must be at least 100 ms");
                    return;
                }
            }
            catch (NumberFormatException notANumber) {
                System.out.println("Connection timeout argument must be an integer value");
                return;
            }
        }
        try {
            this.builder.connectTimeout(Duration.ofMillis(connectionTimeout));
            this.quicConnection = this.builder.build();
            this.quicConnection.connect();
            System.out.println("Ok, connected to " + this.quicConnection.getUri() + "\n");
        }
        catch (IOException e) {
            System.out.println("\nError: " + e);
        }
    }

    private void close(String arg) {
        if (this.quicConnection != null) {
            this.quicConnection.close();
        }
    }

    private void httpGet(String arg) {
        if (this.quicConnection == null) {
            System.out.println("Error: no connected");
            return;
        }
        try {
            if (this.httpClient == null) {
                this.httpClient = KwikCli.createHttpClient(this.httpVersion, (QuicClientConnection)this.quicConnection, false);
            }
            InetSocketAddress serverAddress = this.quicConnection.getServerAddress();
            HttpRequest request = HttpRequest.newBuilder().uri(new URI("https", null, serverAddress.getHostName(), serverAddress.getPort(), arg, null, null)).build();
            Instant start = Instant.now();
            CompletableFuture<HttpResponse<Path>> sendResult = this.httpClient.sendAsync(request, HttpResponse.BodyHandlers.ofFile(this.createNewFile(arg).toPath()));
            CompletionStage processResult = sendResult.thenAccept(response -> {
                String speed;
                Instant done = Instant.now();
                Duration duration = Duration.between(start, done);
                try {
                    long size = Files.size((Path)response.body());
                    speed = String.format("%.2f", Float.valueOf((float)size / (float)duration.toMillis() / 1000.0f));
                }
                catch (IOException e) {
                    speed = "?";
                }
                System.out.println(String.format("Get requested finished in %.2f sec (%s MB/s) : %s", Float.valueOf((float)duration.toMillis() / 1000.0f), speed, response));
            });
            ((CompletableFuture)processResult).exceptionally(error -> {
                System.out.println("Error: " + error);
                return null;
            });
            this.currentHttpGetResult = sendResult;
        }
        catch (IOException | URISyntaxException e) {
            System.out.println("Error: " + e);
        }
    }

    private void httpStop(String arg) {
        this.currentHttpGetResult.cancel(true);
    }

    private File createNewFile(String baseName) throws IOException {
        File file;
        if (baseName.startsWith("/") && baseName.length() > 1) {
            baseName = baseName.substring(1);
        }
        if (!(file = new File((baseName = baseName.replace('/', '_')) + ".dat")).exists()) {
            return file;
        }
        for (int i = 0; i < 1000; ++i) {
            file = new File(baseName + i + ".dat");
            if (file.exists()) continue;
            return file;
        }
        throw new IOException("Cannot create output file '" + baseName + ".dat'");
    }

    private void newConnectionIds(String args) {
        int newConnectionIdCount = 1;
        int retirePriorTo = 0;
        if (!args.isEmpty()) {
            try {
                Object[] intArgs = Stream.of(args.split(" +")).map(arg -> Integer.parseInt(arg)).toArray();
                newConnectionIdCount = (Integer)intArgs[0];
                if (intArgs.length > 1) {
                    retirePriorTo = (Integer)intArgs[1];
                }
            }
            catch (NumberFormatException notANumber) {
                System.out.println("Expected arguments: [<number of new ids>] [<sequence number to retire cids prior to>]");
                return;
            }
        }
        byte[][] newConnectionIds = this.quicConnection.newConnectionIds(newConnectionIdCount, retirePriorTo);
        System.out.println("Generated new (source) connection id's: " + Arrays.stream(newConnectionIds).map(cid -> ByteUtils.bytesToHex((byte[])cid)).collect(Collectors.joining(", ")));
    }

    private void printConnectionIds(String arg) {
        System.out.println("Source (client) connection id's:");
        this.quicConnection.getSourceConnectionIds().entrySet().stream().sorted(Map.Entry.comparingByKey()).forEach(entry -> System.out.println(this.toString(((ConnectionIdInfo)entry.getValue()).getConnectionIdStatus()) + " " + entry.getKey() + ": " + ByteUtils.bytesToHex((byte[])((ConnectionIdInfo)entry.getValue()).getConnectionId())));
        System.out.println("Destination (server) connection id's:");
        this.quicConnection.getDestinationConnectionIds().entrySet().stream().sorted(Map.Entry.comparingByKey()).forEach(entry -> System.out.println(this.toString(((ConnectionIdInfo)entry.getValue()).getConnectionIdStatus()) + " " + entry.getKey() + ": " + ByteUtils.bytesToHex((byte[])((ConnectionIdInfo)entry.getValue()).getConnectionId())));
    }

    private String toString(ConnectionIdStatus connectionIdStatus) {
        switch (connectionIdStatus) {
            case NEW: {
                return " ";
            }
            case IN_USE: {
                return "*";
            }
            case USED: {
                return ".";
            }
            case RETIRED: {
                return "x";
            }
        }
        throw new RuntimeException("");
    }

    private void nextDestinationConnectionId(String arg) {
        byte[] newConnectionId = this.quicConnection.nextDestinationConnectionId();
        if (newConnectionId != null) {
            System.out.println("Switched to next destination connection id: " + ByteUtils.bytesToHex((byte[])newConnectionId));
        } else {
            System.out.println("Cannot switch to next destination connect id, because there is none available");
        }
    }

    private void retireConnectionId(String arg) {
        this.quicConnection.retireDestinationConnectionId(this.toInt(arg));
    }

    private void changeUdpPort(String args) {
        this.quicConnection.changeAddress();
    }

    private void help(String arg) {
        System.out.println("available commands: " + this.commands.keySet().stream().collect(Collectors.joining(", ")));
    }

    private void quit(String arg) {
        this.running = false;
    }

    private void unknown(String cmd) {
        System.out.println("unknown command: " + cmd);
    }

    private void sendPing(String arg) {
        this.quicConnection.ping();
    }

    private void printClientParams(String arg) {
        System.out.print("Client transport parameters: \n" + this.params + "\n");
    }

    private void setClientParameter(String argLine) {
        String[] args = argLine.split("\\s+");
        if (args.length == 2) {
            String name = args[0];
            String value = args[1];
            this.setClientParameter(name, value);
        } else {
            System.out.println("Incorrect parameters; should be <transport parameter name> <value>.");
            System.out.println("Supported parameters: ");
            this.printSupportedParameters();
        }
    }

    private void printSupportedParameters() {
        System.out.println("- idle (max idle timeout in seconds)");
        System.out.println("- cids (active connection id limit)");
        System.out.println("- maxstreamdata (receive buffer size)");
        System.out.println("- maxuni (max peer initiated unidirectional streams)");
        System.out.println("- maxbidi (max peer initiated bidirectional streams)");
        System.out.println("- payload (max udp payload)");
    }

    private void setClientParameter(String name, String value) {
        switch (name) {
            case "idle": {
                this.builder.maxIdleTimeout(Duration.ofSeconds(this.toInt(value).intValue()));
                this.params.maxIdleTimeout = this.toInt(value);
                break;
            }
            case "cids": {
                this.builder.activeConnectionIdLimit(this.toInt(value).intValue());
                this.params.activeConnectionIdLimit = this.toInt(value);
                break;
            }
            case "maxStreamData": 
            case "maxstreamdata": {
                this.builder.defaultStreamReceiveBufferSize(this.toLong(value));
                this.params.defaultStreamReceiveBufferSize = this.toLong(value);
                break;
            }
            case "maxuni": 
            case "maxUni": {
                this.builder.maxOpenPeerInitiatedUnidirectionalStreams(this.toInt(value).intValue());
                this.params.maxOpenPeerInitiatedUnidirectionalStreams = this.toInt(value);
                break;
            }
            case "maxbidi": 
            case "maxBidi": {
                this.builder.maxOpenPeerInitiatedBidirectionalStreams(this.toInt(value).intValue());
                this.params.maxOpenPeerInitiatedBidirectionalStreams = this.toInt(value);
                break;
            }
            case "payload": {
                this.builder.maxUdpPayloadSize(this.toInt(value).intValue());
                this.params.maxUdpPayloadSize = this.toInt(value);
                if (this.toInt(value) <= 1500) break;
                System.out.println(String.format("Warning: client will read at most %d datagram bytes", 1500));
                break;
            }
            default: {
                System.out.println("Parameter must be one of:");
                this.printSupportedParameters();
            }
        }
    }

    private void printServerParams(String arg) {
        if (this.quicConnection != null) {
            TransportParameters parameters = this.quicConnection.getPeerTransportParameters();
            System.out.println("Server transport parameters: " + parameters);
        } else {
            System.out.println("Server transport parameters still unknown (no connection)");
        }
    }

    private void printStatistics(String arg) {
        if (this.quicConnection != null) {
            System.out.println(this.quicConnection.getStats());
        }
    }

    private void setScidLength(String arg) {
        this.builder.connectionIdLength(this.toInt(arg).intValue());
    }

    private void updateKeys(String arg) {
        this.quicConnection.updateKeys();
        this.quicConnection.ping();
    }

    private void error(Exception error) {
        System.out.println("error: " + error);
        error.printStackTrace();
    }

    private void prompt() {
        System.out.print("> ");
        System.out.flush();
    }

    private Integer toInt(String value) {
        try {
            return Integer.parseInt(value);
        }
        catch (NumberFormatException e) {
            System.out.println("Error: value not an integer; using 0");
            return 0;
        }
    }

    private Long toLong(String value) {
        try {
            return Long.parseLong(value);
        }
        catch (NumberFormatException e) {
            System.out.println("Error: value not an integer; using 0");
            return 0L;
        }
    }

    private class ClientParameters {
        public Integer maxIdleTimeout = 60;
        public Integer activeConnectionIdLimit = 2;
        public Long defaultStreamReceiveBufferSize = 250000L;
        public Integer maxOpenPeerInitiatedUnidirectionalStreams = 3;
        public Integer maxOpenPeerInitiatedBidirectionalStreams = 3;
        public Integer maxUdpPayloadSize = 1500;

        private ClientParameters() {
        }

        public String toString() {
            return "idle=" + this.maxIdleTimeout + " (seconds)\ncids=" + this.activeConnectionIdLimit + "\nmaxStreamData=" + this.defaultStreamReceiveBufferSize + "\nmaxUni=" + this.maxOpenPeerInitiatedUnidirectionalStreams + "\nmaxBidi=" + this.maxOpenPeerInitiatedBidirectionalStreams + "\npayload=" + this.maxUdpPayloadSize;
        }
    }
}

