/*
 * Decompiled with CFR 0.152.
 */
package tech.kwik.cli;

import java.io.ByteArrayInputStream;
import java.io.File;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.PrintStream;
import java.lang.reflect.Constructor;
import java.lang.reflect.InvocationTargetException;
import java.net.InetSocketAddress;
import java.net.MalformedURLException;
import java.net.URI;
import java.net.URISyntaxException;
import java.net.URL;
import java.net.http.HttpClient;
import java.net.http.HttpRequest;
import java.net.http.HttpResponse;
import java.nio.charset.Charset;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.nio.file.StandardOpenOption;
import java.security.KeyFactory;
import java.security.KeyStore;
import java.security.KeyStoreException;
import java.security.NoSuchAlgorithmException;
import java.security.PrivateKey;
import java.security.UnrecoverableKeyException;
import java.security.cert.Certificate;
import java.security.cert.CertificateEncodingException;
import java.security.cert.CertificateException;
import java.security.cert.CertificateFactory;
import java.security.cert.X509Certificate;
import java.security.spec.InvalidKeySpecException;
import java.security.spec.PKCS8EncodedKeySpec;
import java.time.Duration;
import java.time.Instant;
import java.util.Arrays;
import java.util.Base64;
import java.util.List;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import java.util.stream.Collectors;
import javax.net.ssl.KeyManager;
import javax.net.ssl.KeyManagerFactory;
import javax.net.ssl.X509ExtendedKeyManager;
import org.apache.commons.cli.CommandLine;
import org.apache.commons.cli.DefaultParser;
import org.apache.commons.cli.HelpFormatter;
import org.apache.commons.cli.Option;
import org.apache.commons.cli.Options;
import org.apache.commons.cli.ParseException;
import tech.kwik.agent15.TlsConstants;
import tech.kwik.cli.InteractiveShell;
import tech.kwik.core.KwikVersion;
import tech.kwik.core.QuicClientConnection;
import tech.kwik.core.QuicConnection;
import tech.kwik.core.QuicSessionTicket;
import tech.kwik.core.impl.QuicClientConnectionImpl;
import tech.kwik.core.impl.QuicSessionTicketImpl;
import tech.kwik.core.impl.VersionNegotiationFailure;
import tech.kwik.core.log.FileLogger;
import tech.kwik.core.log.Logger;
import tech.kwik.core.log.SysOutLogger;
import tech.kwik.h09.client.Http09Client;

public class KwikCli {
    private static String DEFAULT_LOG_ARGS = "wip";
    private static Options cmdLineOptions;
    private String newSessionTicketsFilename;
    private boolean useZeroRtt;
    private String serverCertificatesFile;
    private int keepAliveTime;
    private HttpVersion httpVersion;
    private String alpn;
    private Logger logger;

    public void run(String[] rawArgs) throws Exception {
        CommandLine cmd = this.getCommandLine(rawArgs);
        boolean interactiveMode = cmd.hasOption("i");
        QuicClientConnection.Builder connectionBuilder = this.createConnectionBuilder(interactiveMode);
        String httpRequestPath = this.processUrlArgs(cmd, connectionBuilder);
        if (cmd.hasOption("preferIPv6")) {
            connectionBuilder.preferIPv6();
        }
        this.processCipherArgs(cmd, connectionBuilder);
        if (cmd.hasOption("noCertificateCheck")) {
            connectionBuilder.noServerCertificateCheck();
        }
        if (cmd.hasOption("trustStore")) {
            String password = cmd.hasOption("trustStorePassword") ? cmd.getOptionValue("trustStorePassword") : "";
            connectionBuilder.customTrustStore(KeyStore.getInstance(new File(cmd.getOptionValue("trustStore")), password.toCharArray()));
        }
        if (cmd.hasOption("saveServerCertificates")) {
            this.serverCertificatesFile = cmd.getOptionValue("saveServerCertificates");
        }
        this.processLoggerArgs(cmd, connectionBuilder);
        this.processVersionArgs(cmd, connectionBuilder);
        this.httpVersion = this.determineHttpVersion();
        this.alpn = this.determineAlpn(cmd);
        this.processConnectTimeoutArgs(cmd, connectionBuilder);
        this.processKeepAliveArg(cmd);
        httpRequestPath = this.extractHttpRequestPath(cmd, connectionBuilder, httpRequestPath);
        this.useZeroRtt = cmd.hasOption("Z");
        if (this.useZeroRtt && httpRequestPath == null) {
            throw new IllegalArgumentException("Option --use0RTT requires option --http");
        }
        String outputFile = this.extractOutputFile(cmd);
        this.processSecretsArgs(cmd, connectionBuilder);
        this.processSessionTicketSaveArg(cmd, connectionBuilder);
        boolean useSessionTicket = this.processSessionTicketArg(cmd, connectionBuilder);
        if (this.useZeroRtt && !useSessionTicket) {
            throw new IllegalArgumentException("Option --use0RTT requires option --sessionTicket");
        }
        this.processClientCertificateArgs(cmd, connectionBuilder);
        this.processQuantumReadinessTestArg(cmd, connectionBuilder);
        this.processInitialRttArg(cmd, connectionBuilder);
        this.processBufferSizeArg(cmd, connectionBuilder);
        if (this.httpVersion == HttpVersion.HTTP3 && this.useZeroRtt) {
            throw new IllegalArgumentException("Option --use0RTT is not yet supported by this HTTP3 implementation.");
        }
        try {
            if (interactiveMode) {
                new InteractiveShell((QuicClientConnectionImpl.ExtendedBuilder)connectionBuilder, this.alpn, this.httpVersion).start();
            } else {
                this.executeRequest(httpRequestPath, outputFile, connectionBuilder);
                Thread.sleep(1000L);
            }
            System.out.println("Terminating Kwik");
            if (!interactiveMode && httpRequestPath == null && this.keepAliveTime == 0) {
                System.out.println("This was quick, huh? Next time, consider using --http09 or --keepAlive argument.");
            }
        }
        catch (IOException e) {
            System.out.println("Got IO error: " + String.valueOf(e));
        }
        catch (VersionNegotiationFailure e) {
            System.out.println("Client and server could not agree on a compatible QUIC version.");
        }
        catch (Exception e) {
            System.out.println("Error: " + e.getMessage());
        }
    }

    private CommandLine getCommandLine(String[] rawArgs) {
        try {
            DefaultParser parser = new DefaultParser();
            CommandLine cmd = parser.parse(cmdLineOptions, rawArgs);
            if (cmd.hasOption("v")) {
                System.out.println("Kwik version: " + KwikVersion.getVersion());
                System.exit(0);
            }
            if (cmd.getArgList().isEmpty()) {
                throw new IllegalArgumentException("Missing arguments");
            }
            return cmd;
        }
        catch (ParseException argError) {
            throw new IllegalArgumentException("Invalid argument: " + argError.getMessage());
        }
    }

    private QuicClientConnection.Builder createConnectionBuilder(boolean interactiveMode) {
        if (interactiveMode) {
            return QuicClientConnectionImpl.newExtendedBuilder();
        }
        return QuicClientConnection.newBuilder();
    }

    /*
     * Enabled force condition propagation
     * Lifted jumps to return sites
     */
    private String processUrlArgs(CommandLine cmd, QuicClientConnection.Builder builder) {
        String httpRequestPath = null;
        List args = cmd.getArgList();
        if (args.size() == 1) {
            String arg = (String)args.get(0);
            try {
                if (arg.startsWith("http://") || arg.startsWith("https://")) {
                    try {
                        URL url = new URL(arg);
                        builder.uri(url.toURI());
                        if (url.getPath().isEmpty()) return httpRequestPath;
                        return url.getPath();
                    }
                    catch (MalformedURLException e) {
                        throw new IllegalArgumentException("Cannot parse URL '" + arg + "'");
                    }
                }
                if (arg.contains(":")) {
                    builder.uri(new URI("//" + arg));
                    return httpRequestPath;
                }
                if (arg.matches("\\d+")) {
                    throw new IllegalArgumentException("Invalid hostname (did you forget to specify an option argument?).");
                }
                builder.uri(new URI("//" + arg + ":443"));
                return httpRequestPath;
            }
            catch (URISyntaxException invalidUri) {
                throw new IllegalArgumentException("Cannot parse URI '" + arg + "'");
            }
        } else if (args.size() == 2) {
            try {
                builder.uri(new URI("//" + (String)args.get(0) + ":" + (String)args.get(1)));
                return httpRequestPath;
            }
            catch (URISyntaxException invalidUri) {
                throw new IllegalArgumentException("Cannot parse URI '" + args.stream().collect(Collectors.joining(":")) + "'");
            }
        } else {
            if (args.size() <= 2) return httpRequestPath;
            throw new IllegalArgumentException("Too many arguments");
        }
    }

    private QuicClientConnection.Builder processCipherArgs(CommandLine cmd, QuicClientConnection.Builder builder) {
        List<String> cipherOpts = List.of("aes128gcm", "aes256gcm", "chacha20");
        List cipherOptions = Arrays.stream(cmd.getOptions()).filter(option -> option.hasLongOpt()).filter(option -> cipherOpts.contains(option.getLongOpt())).distinct().collect(Collectors.toList());
        for (Option cipherOption : cipherOptions) {
            if (cipherOption.getLongOpt().equals("aes128gcm")) {
                builder.cipherSuite(TlsConstants.CipherSuite.TLS_AES_128_GCM_SHA256);
            }
            if (cipherOption.getLongOpt().equals("aes256gcm")) {
                builder.cipherSuite(TlsConstants.CipherSuite.TLS_AES_256_GCM_SHA384);
            }
            if (!cipherOption.getLongOpt().equals("chacha20")) continue;
            builder.cipherSuite(TlsConstants.CipherSuite.TLS_CHACHA20_POLY1305_SHA256);
        }
        return builder;
    }

    private void processLoggerArgs(CommandLine cmd, QuicClientConnection.Builder builder) {
        if (cmd.hasOption("L")) {
            String logFilename = cmd.getOptionValue("L");
            try {
                this.logger = new FileLogger(new File(logFilename));
            }
            catch (IOException fileError) {
                System.err.println("Error: cannot open log file '" + logFilename + "'");
            }
        }
        if (this.logger == null) {
            this.logger = new SysOutLogger();
        }
        String logArg = DEFAULT_LOG_ARGS;
        if (cmd.hasOption('l')) {
            logArg = cmd.getOptionValue('l', logArg);
        }
        if (!logArg.contains("n")) {
            if (logArg.contains("R")) {
                this.logger.logRaw(true);
            }
            if (logArg.contains("r")) {
                this.logger.logRecovery(true);
            }
            if (logArg.contains("c")) {
                this.logger.logCongestionControl(true);
            }
            if (logArg.contains("d")) {
                this.logger.logDecrypted(true);
            }
            if (logArg.contains("S")) {
                this.logger.logSecrets(true);
            }
            if (logArg.contains("p")) {
                this.logger.logPackets(true);
            }
            if (logArg.contains("i")) {
                this.logger.logInfo(true);
            }
            if (logArg.contains("w")) {
                this.logger.logWarning(true);
            }
            if (logArg.contains("s")) {
                this.logger.logStats(true);
            }
            if (logArg.contains("D")) {
                this.logger.logDebug(true);
            }
        }
        if (cmd.hasOption("T")) {
            this.logger.useRelativeTime(true);
        }
        builder.logger(this.logger);
    }

    private void processVersionArgs(CommandLine cmd, QuicClientConnection.Builder builder) {
        QuicConnection.QuicVersion quicVersion = QuicConnection.QuicVersion.V1;
        QuicConnection.QuicVersion preferredVersion = null;
        if (cmd.hasOption("v1v2")) {
            quicVersion = QuicConnection.QuicVersion.V1;
            preferredVersion = QuicConnection.QuicVersion.V2;
        } else if (cmd.hasOption("v2")) {
            quicVersion = QuicConnection.QuicVersion.V2;
        } else if (cmd.hasOption("v1")) {
            quicVersion = QuicConnection.QuicVersion.V1;
        }
        builder.version(quicVersion);
        builder.preferredVersion(preferredVersion);
    }

    private HttpVersion determineHttpVersion() {
        HttpVersion httpVersion = KwikCli.loadHttp3ClientClass() ? HttpVersion.HTTP3 : HttpVersion.HTTP09;
        return httpVersion;
    }

    private String determineAlpn(CommandLine cmd) {
        String alpn;
        if (cmd.hasOption("A")) {
            alpn = cmd.getOptionValue("A", null);
            if (alpn == null) {
                throw new IllegalArgumentException("Missing argument for option -A");
            }
        } else {
            alpn = this.httpVersion == HttpVersion.HTTP3 ? "h3" : "hq-interop";
        }
        return alpn;
    }

    private void processConnectTimeoutArgs(CommandLine cmd, QuicClientConnection.Builder builder) {
        if (cmd.hasOption("c")) {
            try {
                int connectionTimeout = Integer.parseInt(cmd.getOptionValue("c"));
                builder.connectTimeout(Duration.ofSeconds(connectionTimeout));
            }
            catch (NumberFormatException e) {
                throw new IllegalArgumentException("Invalid value for --connectionTimeout: " + cmd.getOptionValue("c"));
            }
        }
    }

    private void processKeepAliveArg(CommandLine cmd) {
        if (cmd.hasOption("k")) {
            try {
                this.keepAliveTime = Integer.parseInt(cmd.getOptionValue("k"));
            }
            catch (NumberFormatException e) {
                throw new IllegalArgumentException("Invalid value for --keepAlive: " + cmd.getOptionValue("k"));
            }
        }
    }

    private String extractHttpRequestPath(CommandLine cmd, QuicClientConnection.Builder builder, String defaultValue) {
        Object httpRequestPath = defaultValue;
        if (cmd.hasOption("H")) {
            if (cmd.getOptionValue("H") == null) {
                throw new IllegalArgumentException("Missing argument for option -H");
            }
            httpRequestPath = cmd.getOptionValue("H");
            if (!((String)httpRequestPath).startsWith("/")) {
                httpRequestPath = "/" + (String)httpRequestPath;
            }
        }
        return httpRequestPath;
    }

    private String extractOutputFile(CommandLine cmd) {
        String outputFile = null;
        if (cmd.hasOption("O")) {
            outputFile = cmd.getOptionValue("O");
            if (outputFile == null) {
                throw new IllegalArgumentException("Missing argument for option -O");
            }
            if (Files.exists(Paths.get(outputFile, new String[0]), new LinkOption[0]) && !Files.isWritable(Paths.get(outputFile, new String[0]))) {
                throw new IllegalArgumentException("Output file '" + outputFile + "' is not writable.");
            }
        }
        return outputFile;
    }

    private void processSecretsArgs(CommandLine cmd, QuicClientConnection.Builder builder) {
        if (cmd.hasOption("secrets")) {
            String secretsFile = cmd.getOptionValue("secrets");
            if (secretsFile == null) {
                throw new IllegalArgumentException("Missing argument for option -secrets");
            }
            if (Files.exists(Paths.get(secretsFile, new String[0]), new LinkOption[0]) && !Files.isWritable(Paths.get(secretsFile, new String[0]))) {
                throw new IllegalArgumentException("Secrets file '" + secretsFile + "' is not writable.");
            }
            builder.secrets(Paths.get(secretsFile, new String[0]));
        }
    }

    private void processSessionTicketSaveArg(CommandLine cmd, QuicClientConnection.Builder builder) {
        if (cmd.hasOption("S")) {
            this.newSessionTicketsFilename = cmd.getOptionValue("S");
            if (this.newSessionTicketsFilename == null) {
                throw new IllegalArgumentException("Missing argument for option -S");
            }
        }
    }

    private boolean processSessionTicketArg(CommandLine cmd, QuicClientConnection.Builder builder) {
        if (cmd.hasOption("R")) {
            String sessionTicketFile = cmd.getOptionValue("R");
            if (sessionTicketFile == null) {
                throw new IllegalArgumentException("Missing argument for option -R");
            }
            if (!Files.isReadable(Paths.get(sessionTicketFile, new String[0]))) {
                throw new IllegalArgumentException("Session ticket file '" + sessionTicketFile + "' is not readable.");
            }
            try {
                byte[] ticketData = Files.readAllBytes(Paths.get(sessionTicketFile, new String[0]));
                QuicSessionTicketImpl sessionTicket = QuicSessionTicketImpl.deserialize((byte[])ticketData);
                builder.sessionTicket((QuicSessionTicket)sessionTicket);
                return true;
            }
            catch (IOException e) {
                throw new IllegalArgumentException("Error while reading session ticket file.");
            }
        }
        return false;
    }

    private void processClientCertificateArgs(CommandLine cmd, QuicClientConnection.Builder builder) {
        if (cmd.hasOption("clientCertificate") && cmd.hasOption("keyManager")) {
            throw new IllegalArgumentException("Options --clientCertificate and --keyManager should not be used together");
        }
        if (cmd.hasOption("keyManager")) {
            try {
                String keyStorePassword = cmd.hasOption("keyManagerPassword") ? cmd.getOptionValue("keyManagerPassword") : "";
                KeyStore keyStore = KeyStore.getInstance(new File(cmd.getOptionValue("keyManager")), keyStorePassword.toCharArray());
                KeyManagerFactory keyManagerFactory = KeyManagerFactory.getInstance(KeyManagerFactory.getDefaultAlgorithm());
                keyManagerFactory.init(keyStore, keyStorePassword.toCharArray());
                KeyManager keyManager = keyManagerFactory.getKeyManagers()[0];
                if (keyManager instanceof X509ExtendedKeyManager) {
                    builder.clientKeyManager((X509ExtendedKeyManager)keyManager);
                }
            }
            catch (IOException | KeyStoreException | NoSuchAlgorithmException | UnrecoverableKeyException | CertificateException e) {
                throw new IllegalArgumentException("Error while reading client key manager", e);
            }
        }
        if (cmd.hasOption("clientCertificate") && cmd.hasOption("clientKey")) {
            try {
                builder.clientCertificate(KwikCli.readCertificate(cmd.getOptionValue("clientCertificate")));
                builder.clientCertificateKey(KwikCli.readKey(cmd.getOptionValue("clientKey")));
            }
            catch (Exception e) {
                throw new IllegalArgumentException("Error while reading client certificate or key: " + e.getMessage());
            }
        } else if (cmd.hasOption("clientCertificate") || cmd.hasOption("clientKey")) {
            throw new IllegalArgumentException("Options --clientCertificate and --clientKey should always be used together");
        }
    }

    private void processQuantumReadinessTestArg(CommandLine cmd, QuicClientConnection.Builder builder) {
        if (cmd.hasOption("quantumReadinessTest")) {
            try {
                builder.quantumReadinessTest(Integer.parseInt(cmd.getOptionValue("quantumReadinessTest")));
            }
            catch (NumberFormatException e) {
                throw new IllegalArgumentException("Invalid value for --quantumReadinessTest: " + cmd.getOptionValue("quantumReadinessTest"));
            }
        }
    }

    private void processInitialRttArg(CommandLine cmd, QuicClientConnection.Builder builder) {
        if (cmd.hasOption("initialRtt")) {
            try {
                builder.initialRtt(Integer.parseInt(cmd.getOptionValue("initialRtt")));
            }
            catch (NumberFormatException e) {
                throw new IllegalArgumentException("Invalid value for --initialRtt: " + cmd.getOptionValue("initialRtt"));
            }
        }
    }

    private void processBufferSizeArg(CommandLine cmd, QuicClientConnection.Builder connectionBuilder) {
        if (cmd.hasOption("B")) {
            String sizeSpecification = cmd.getOptionValue("B");
            Matcher matcher = Pattern.compile("(\\d+)([KM])?").matcher(sizeSpecification);
            if (matcher.matches()) {
                long bufferSize;
                int value = Integer.parseInt(matcher.group(1));
                int unit = 1;
                if (matcher.group(2) != null) {
                    if (matcher.group(2).equals("K")) {
                        unit = 1024;
                    } else if (matcher.group(2).equals("M")) {
                        unit = 0x100000;
                    }
                }
                if ((bufferSize = (long)(value * unit)) < 1500L || bufferSize > 0x6400000L) {
                    throw new IllegalArgumentException(String.format("Buffer size must be between %d and 100M.", 1500));
                }
                connectionBuilder.defaultStreamReceiveBufferSize(Long.valueOf(bufferSize));
                System.out.println("Receive buffer size set to " + bufferSize + " bytes.");
            } else {
                throw new IllegalArgumentException("Invalid buffer size specification: " + sizeSpecification);
            }
        }
    }

    private void executeRequest(String httpRequestPath, String outputFile, QuicClientConnection.Builder builder) throws IOException {
        if (this.httpVersion == HttpVersion.HTTP3) {
            builder.applicationProtocol("h3");
        } else {
            builder.applicationProtocol("hq-interop");
        }
        QuicClientConnection quicConnection = builder.build();
        if (httpRequestPath != null) {
            try {
                long size;
                Instant done;
                String response;
                HttpResponse<Object> httpResponse;
                Instant start;
                HttpClient httpClient = KwikCli.createHttpClient(this.httpVersion, quicConnection, this.useZeroRtt);
                InetSocketAddress serverAddress = quicConnection.getServerAddress();
                HttpRequest request = HttpRequest.newBuilder().uri(new URI("https", null, serverAddress.getHostName(), serverAddress.getPort(), httpRequestPath, null, null)).build();
                if (outputFile != null) {
                    if (new File(outputFile).isDirectory()) {
                        outputFile = new File(outputFile, new File(httpRequestPath).getName()).getAbsolutePath();
                    }
                    start = Instant.now();
                    httpResponse = httpClient.send(request, HttpResponse.BodyHandlers.ofFile(Paths.get(outputFile, new String[0])));
                    response = httpResponse.toString();
                    done = Instant.now();
                    size = Files.size((Path)httpResponse.body());
                } else {
                    start = Instant.now();
                    httpResponse = httpClient.send(request, HttpResponse.BodyHandlers.ofString());
                    done = Instant.now();
                    size = httpResponse.body().length();
                    response = httpResponse.toString();
                    try {
                        Thread.sleep(500L);
                    }
                    catch (InterruptedException interruptedException) {
                        // empty catch block
                    }
                    System.out.println("Server returns: \nStatus code " + httpResponse.statusCode() + "\n" + String.valueOf(httpResponse.headers().map()) + "\n" + (String)httpResponse.body());
                }
                Duration duration = Duration.between(start, done);
                String speed = String.format("%.2f", Float.valueOf((float)size / (float)duration.toMillis() / 1000.0f));
                System.out.println(String.format("Get requested finished in %.2f sec (%s MB/s) : %s", Float.valueOf((float)duration.toMillis() / 1000.0f), speed, response));
            }
            catch (InterruptedException interruptedException) {
                System.out.println("HTTP request is interrupted");
            }
            catch (URISyntaxException e) {
                throw new RuntimeException();
            }
        } else {
            quicConnection.connect();
            if (this.keepAliveTime > 0) {
                quicConnection.keepAlive(this.keepAliveTime);
                try {
                    Thread.sleep((long)(this.keepAliveTime + 30) * 1000L);
                }
                catch (InterruptedException interruptedException) {
                    // empty catch block
                }
            }
        }
        if (this.serverCertificatesFile != null) {
            KwikCli.storeServerCertificates(quicConnection, this.serverCertificatesFile);
        }
        if (this.newSessionTicketsFilename != null) {
            KwikCli.storeNewSessionTickets(quicConnection, this.newSessionTicketsFilename);
        }
        quicConnection.close();
    }

    private static boolean loadHttp3ClientClass() {
        try {
            KwikCli.getHttp3ClientClass();
            return true;
        }
        catch (ClassNotFoundException e) {
            return false;
        }
    }

    static HttpClient createHttpClient(HttpVersion httpVersion, QuicClientConnection quicConnection, boolean useZeroRtt) {
        if (httpVersion == HttpVersion.HTTP3) {
            try {
                Class http3ClientClass = KwikCli.getHttp3ClientClass();
                Constructor constructor = http3ClientClass.getConstructor(QuicConnection.class, Duration.class, Long.class);
                long maxReceiveBufferSize = 50000000L;
                Duration connectionTimeout = Duration.ofSeconds(60L);
                HttpClient http3Client = (HttpClient)constructor.newInstance(quicConnection, connectionTimeout, maxReceiveBufferSize);
                return http3Client;
            }
            catch (ClassNotFoundException | IllegalAccessException | InstantiationException | NoSuchMethodException | InvocationTargetException e) {
                throw new RuntimeException(e);
            }
        }
        return new Http09Client(quicConnection, useZeroRtt);
    }

    private static Class getHttp3ClientClass() throws ClassNotFoundException {
        Class<?> http3ClientClass;
        try {
            http3ClientClass = KwikCli.class.getClassLoader().loadClass("net.luminis.http3.Http3SingleConnectionClient");
        }
        catch (ClassNotFoundException e) {
            http3ClientClass = KwikCli.class.getClassLoader().loadClass("tech.kwik.flupke.Http3SingleConnectionClient");
        }
        return http3ClientClass;
    }

    private static PrivateKey readKey(String clientKey) throws IOException, InvalidKeySpecException {
        String key = new String(Files.readAllBytes(Paths.get(clientKey, new String[0])), Charset.defaultCharset());
        if (key.contains("BEGIN PRIVATE KEY")) {
            return KwikCli.loadRSAKey(key);
        }
        if (key.contains("BEGIN EC PRIVATE KEY")) {
            throw new IllegalArgumentException("EC private key must be in DER format");
        }
        return KwikCli.loadECKey(Files.readAllBytes(Paths.get(clientKey, new String[0])));
    }

    private static PrivateKey loadRSAKey(String key) throws InvalidKeySpecException {
        String privateKeyPEM = key.replace("-----BEGIN PRIVATE KEY-----", "").replaceAll(System.lineSeparator(), "").replace("-----END PRIVATE KEY-----", "");
        byte[] encoded = Base64.getMimeDecoder().decode(privateKeyPEM);
        try {
            KeyFactory keyFactory = KeyFactory.getInstance("RSA");
            PKCS8EncodedKeySpec keySpec = new PKCS8EncodedKeySpec(encoded);
            return keyFactory.generatePrivate(keySpec);
        }
        catch (NoSuchAlgorithmException e) {
            throw new RuntimeException("Missing key algorithm RSA");
        }
    }

    private static PrivateKey loadECKey(byte[] pkcs8key) throws InvalidKeySpecException {
        try {
            PKCS8EncodedKeySpec spec = new PKCS8EncodedKeySpec(pkcs8key);
            KeyFactory factory = KeyFactory.getInstance("EC");
            PrivateKey privateKey = factory.generatePrivate(spec);
            return privateKey;
        }
        catch (NoSuchAlgorithmException e) {
            throw new RuntimeException("Missing ECDSA support");
        }
    }

    private static X509Certificate readCertificate(String certificateFile) throws IOException, CertificateException {
        String fileContent = new String(Files.readAllBytes(Paths.get(certificateFile, new String[0])), Charset.defaultCharset());
        CertificateFactory certificateFactory = CertificateFactory.getInstance("X.509");
        if (fileContent.startsWith("-----BEGIN CERTIFICATE-----")) {
            String encodedCertificate = fileContent.replace("-----BEGIN CERTIFICATE-----", "").replaceAll(System.lineSeparator(), "").replace("-----END CERTIFICATE-----", "");
            Certificate certificate = certificateFactory.generateCertificate(new ByteArrayInputStream(Base64.getDecoder().decode(encodedCertificate)));
            return (X509Certificate)certificate;
        }
        throw new IllegalArgumentException("Invalid certificate file");
    }

    private static void storeServerCertificates(QuicClientConnection quicConnection, String serverCertificatesFile) throws IOException {
        List serverCertificateChain = quicConnection.getServerCertificateChain();
        if (!((String)serverCertificatesFile).endsWith(".pem")) {
            serverCertificatesFile = (String)serverCertificatesFile + ".pem";
        }
        PrintStream out = new PrintStream(new FileOutputStream(new File((String)serverCertificatesFile)));
        for (X509Certificate cert : serverCertificateChain) {
            out.println("-----BEGIN CERTIFICATE-----");
            try {
                out.print(new String(Base64.getMimeEncoder().encode(cert.getEncoded())));
            }
            catch (CertificateEncodingException e) {
                throw new IOException(e.getMessage());
            }
            out.println("\n-----END CERTIFICATE-----");
            out.println("\n");
        }
        out.close();
    }

    private static void storeNewSessionTickets(QuicClientConnection quicConnection, String baseFilename) {
        if (quicConnection.getNewSessionTickets().isEmpty()) {
            try {
                Thread.sleep(500L);
            }
            catch (InterruptedException interruptedException) {
                // empty catch block
            }
            if (quicConnection.getNewSessionTickets().isEmpty()) {
                System.out.println("There are no new session tickets to store.");
            }
        }
        quicConnection.getNewSessionTickets().stream().forEach(ticket -> KwikCli.storeNewSessionTicket(ticket, baseFilename));
    }

    private static void storeNewSessionTicket(QuicSessionTicket ticket, String baseFilename) {
        int i;
        int maxFiles = 100;
        File savedSessionTicket = new File(baseFilename + ".bin");
        for (i = 1; i <= maxFiles && savedSessionTicket.exists(); ++i) {
            savedSessionTicket = new File(baseFilename + i + ".bin");
        }
        if (i > maxFiles) {
            System.out.println("Cannot store ticket: too many files with base name '" + baseFilename + "' already exist.");
            return;
        }
        try {
            Files.write(savedSessionTicket.toPath(), ticket.serialize(), StandardOpenOption.CREATE);
        }
        catch (IOException e) {
            System.err.println("Saving new session ticket failed: " + String.valueOf(e));
        }
    }

    public static void usage() {
        HelpFormatter helpFormatter = new HelpFormatter();
        helpFormatter.setWidth(79);
        helpFormatter.printHelp("kwik <host>:<port> OR kwik <host> <port> \tOR kwik http[s]://host:port[/path]", cmdLineOptions);
    }

    public static void main(String[] args) throws Exception {
        try {
            KwikCli.createCommandLineOptions();
            new KwikCli().run(args);
        }
        catch (IllegalArgumentException wrongArgs) {
            System.out.println("Incorrect command: " + wrongArgs.getMessage());
            KwikCli.usage();
        }
    }

    private static void createCommandLineOptions() {
        cmdLineOptions = new Options();
        cmdLineOptions.addOption("l", "log", true, "logging options: [pdrcsiRSD]: (p)ackets received/sent, (d)ecrypted bytes, (r)ecovery, (c)ongestion control, (s)tats, (i)nfo, (w)arning, (R)aw bytes, (S)ecrets, (D)ebug;  default is \"" + DEFAULT_LOG_ARGS + "\", use (n)one to disable");
        cmdLineOptions.addOption("h", "help", false, "show help");
        cmdLineOptions.addOption("v1", "use Quic version 1");
        cmdLineOptions.addOption("v2", "use Quic version 2");
        cmdLineOptions.addOption("v1v2", "use Quic version 1, request version 2");
        cmdLineOptions.addOption("A", "alpn", true, "set alpn (interactive mode only)");
        cmdLineOptions.addOption("R", "resumption key", true, "session ticket file");
        cmdLineOptions.addOption("c", "connectionTimeout", true, "connection timeout in seconds");
        cmdLineOptions.addOption("i", "interactive", false, "start interactive shell");
        cmdLineOptions.addOption("k", "keepAlive", true, "connection keep alive time in seconds");
        cmdLineOptions.addOption("L", "logFile", true, "file to write log message to");
        cmdLineOptions.addOption("O", "output", true, "write server response to file");
        cmdLineOptions.addOption("H", "http", true, "send HTTP GET request, arg is path, e.g. '/index.html'");
        cmdLineOptions.addOption("S", "storeTickets", true, "basename of file to store new session tickets");
        cmdLineOptions.addOption("T", "relativeTime", false, "log with time (in seconds) since first packet");
        cmdLineOptions.addOption("Z", "use0RTT", false, "use 0-RTT if possible (requires -H)");
        cmdLineOptions.addOption(null, "secrets", true, "write secrets to file (Wireshark format)");
        cmdLineOptions.addOption("v", "version", false, "show Kwik version");
        cmdLineOptions.addOption(null, "initialRtt", true, "custom initial RTT value (default is 500)");
        cmdLineOptions.addOption(null, "chacha20", false, "use ChaCha20 as only cipher suite");
        cmdLineOptions.addOption(null, "noCertificateCheck", false, "do not check server certificate");
        cmdLineOptions.addOption(null, "saveServerCertificates", true, "store server certificates in given file");
        cmdLineOptions.addOption(null, "quantumReadinessTest", true, "add number of random bytes to client hello");
        cmdLineOptions.addOption(null, "clientCertificate", true, "certificate (file) for client authentication");
        cmdLineOptions.addOption(null, "clientKey", true, "private key (file) for client certificate");
        cmdLineOptions.addOption(null, "chacha20", false, "use ChaCha20 cipher suite");
        cmdLineOptions.addOption(null, "aes128gcm", false, "use AEAD_AES_128_GCM cipher suite");
        cmdLineOptions.addOption(null, "aes256gcm", false, "use AEAD_AES_256_GCM cipher suite");
        cmdLineOptions.addOption(null, "trustStore", true, "use custom trust store (to use non default CA's)");
        cmdLineOptions.addOption(null, "trustStorePassword", true, "password for custom trust store");
        cmdLineOptions.addOption(null, "keyManager", true, "client authentication key manager");
        cmdLineOptions.addOption(null, "keyManagerPassword", true, "password for client authentication key manager and key password");
        cmdLineOptions.addOption(null, "preferIPv6", false, "use IPv6 address if available");
        cmdLineOptions.addOption("B", "receiveBuffer", true, String.format("receive buffer size, e.g. \"500K\" or \"5M\" (default is %dK)", 244L));
    }

    public static enum HttpVersion {
        HTTP09,
        HTTP3;

    }
}

