package tech.greenfield.vertx.irked;

import java.util.*;

import io.vertx.core.MultiMap;
import io.vertx.core.http.HttpServerResponse;
import io.vertx.core.http.impl.headers.HeadersMultiMap;
import io.vertx.ext.web.RoutingContext;
import tech.greenfield.vertx.irked.status.HttpStatuses;
import tech.greenfield.vertx.irked.status.InternalServerError;
import tech.greenfield.vertx.irked.status.OK;

/**
 * Top class representing HTTP response headers in Irked.
 * This class is mostly useful as a quick way to abort processing with the correct HTTP error response, as it
 * extends {@linkplain Exception}, but can also be used with the various {@code Request.send()} methods as a
 * shorthand for calling {@code response().setResponseCode(...).setStatusMessage(...)}
 * @author odeda
 */
public class HttpError extends Exception {

	/**
	 * A RuntimeException wrapper for HttpError to allow it to be thrown from within non-declaring code. It is
	 * meant to be used in Vert.x Future handler lambdas.
	 * This is the type generated by {@link HttpError#unchecked()}.
	 * @author odeda
	 */
	public class UncheckedHttpError extends RuntimeException {
		private static final long serialVersionUID = 1L;
		private UncheckedHttpError() {
			super(HttpError.this);
		}
	}

	private static final long serialVersionUID = -7084405660609573926L;
	
	private int statusCode;
	private String statusText;
	private HeadersMultiMap headers = HeadersMultiMap.httpHeaders();
	
	/**
	 * Creates a new HttpError instance with the specified status code and status text.
	 * It is preferable and easier to use the specific class for the standard status code that you want send, so
	 * only use this constructor for non-standard custom response status codes
	 * @param statusCode HTTP status code
	 * @param statusText HTTP status message
	 */
	public HttpError(int statusCode, String statusText) {
		super(statusText);
		this.statusCode = statusCode;
		this.statusText = statusText;
	}
	
	/**
	 * Creates a new HttpError instance with the specified status code and status text and a custom error message
	 * to be serialized out in default JSON serialization.
	 * It is preferable and easier to use the specific class for the standard status code that you want send, so
	 * only use this constructor for non-standard custom response status codes
	 * @param statusCode HTTP status code
	 * @param statusText HTTP status message
	 * @param message The error message to show in the body of the response
	 */
	public HttpError(int statusCode, String statusText, String message) {
		super(message);
		this.statusCode = statusCode;
		this.statusText = statusText;
	}
	
	/**
	 * Creates a new HttpError instance with the specified status code and status text and a causing exception.
	 * The resulting exception will serialize the cause's message as the error message in default JSON serialization.
	 * It is preferable and easier to use the specific class for the standard status code that you want send, so
	 * only use this constructor for non-standard custom response status codes
	 * @param statusCode HTTP status code
	 * @param statusText HTTP status message
	 * @param throwable The exception causing this status to be sent
	 */
	public HttpError(int statusCode, String statusText, Throwable throwable) {
		super(statusText, throwable);
		this.statusCode = statusCode;
		this.statusText = statusText;
	}
	
	/**
	 * Creates a new HttpError instance with the specified status code and status text and a causing exception.
	 * The resulting exception will serialize the provided custom message as the error message in default JSON
	 * serialization, instead of the cause's message.
	 * It is preferable and easier to use the specific class for the standard status code that you want send, so
	 * only use this constructor for non-standard custom response status codes
	 * @param statusCode HTTP status code
	 * @param statusText HTTP status message
	 * @param message The error message to show in the body of the response
	 * @param throwable The exception causing this status to be sent
	 */
	public HttpError(int statusCode, String statusText, String message, Throwable throwable) {
		super(message, throwable);
		this.statusCode = statusCode;
		this.statusText = statusText;
	}
	
	/**
	 * Retrieve this instance's numeric HTTP status code
	 * @return HTTP status code
	 */
	public int getStatusCode() {
		return statusCode;
	}
	
	/**
	 * Retrieve this instance's HTTP status message
	 * @return HTTP status message
	 */
	public String getStatusText() {
		return statusText;
	}
	
	/**
	 * Update the current instance's status message with a non-default HTTP status message.
	 * The use of this method is not recommended and is offered only for supporting Vert.x {@link HttpServerResponse#setStatusMessage(String)}
	 * API.
	 * @param reasonText The non-standard HTTP status message to use
	 * @return the same instance for use as a fluent API
	 */
	public HttpError setStatusText(String reasonText) {
		statusText = reasonText;
		return this;
	}
	
	/**
	 * Check whether this instance represents a 2xx class HTTP response
	 * @return whether the HTTP status code is between 200-299 (inclusive)
	 */
	public boolean isOK() {
		return statusCode / 100 == 2;
	}
	
	/**
	 * Check whether this instance represents a 3xx class HTTP response
	 * @return whether the HTTP status code is between 300-399 (inclusive)
	 */
	public boolean isRedirect() {
		return statusCode / 100 == 3;
	}
	
	/**
	 * Check whether this instance represents a 4xx class HTTP response
	 * @return whether the HTTP status code is between 400-499 (inclusive)
	 */
	public boolean isClientError() {
		return statusCode / 100 == 4;
	}
	
	/**
	 * Check whether this instance represents a 5xx class HTTP response
	 * @return whether the HTTP status code is between 500-599 (inclusive)
	 */
	public boolean isServerError() {
		return statusCode / 100 == 5;
	}
	
	/**
	 * Check whether this instance represents a "error class" HTTP response
	 * @return whether the HTTP status code is between 400-599 (inclusive)
	 */
	public boolean isError() {
		return isClientError() || isServerError();
	}
	
	/**
	 * Add a header line to be written to the response's header by Irked when this status is used to send a response.
	 * @param header header name
	 * @param value header value
	 * @return the same instance for use as a fluent API
	 */
	public HttpError addHeader(String header, String value) {
		this.headers.add(header, value);
		return this;
	}
	
	/**
	 * Retrieve a reference to the current set of additional headers.
	 * @return the current set of additional headers
	 */
	public MultiMap getHeaders() {
		return headers;
	}
	
	/**
	 * Alias to {@link #unchecked()}
	 * @return unchecked {@link RuntimeException} wrapping this status instance
	 */
	public RuntimeException uncheckedWrap() {
		return new UncheckedHttpError();
	}
	
	/**
	 * Helper method to make it easier to throw HTTP statuses out of lambdas.
	 * Outside the lambda you should catch a {@link RuntimeException} and use
	 * {@link HttpError#unwrap(Throwable)} to get the original exception
	 * @return unchecked {@link RuntimeException} wrapping this status instance
	 */
	public RuntimeException unchecked() {
		return new UncheckedHttpError();
	}
	
	/**
	 * Unwrap {@link RuntimeException} wrappers around a logical exception
	 * (hopefully an instance of HttpError)
	 * @param t Throwable to unwrap
	 * @return the first non RuntimeException found
	 */
	public static Throwable unwrap(Throwable t) {
		for (var ex = t; ex != null; ex = ex.getCause())
			if (ex instanceof HttpError)
				return ex;
		// No HTTP status wrapper found, just unwrap runtimes if relevant
		for (var ex = t; ex != null; ex = ex.getCause())
			if (!(ex instanceof RuntimeException))
				return ex;
		// In case we have a chain of runtime exceptions, just return the top wrapper
		return t;
	}
	
	/**
	 * Helper method for OnFail handlers to locate a wrapped HttpError or
	 * create an InternalServerError from an unexpected exception (whether it
	 * is wrapped or not)
	 * @param t Throwable to be analyzed
	 * @return the wrapped HttpError instance or a new {@link InternalServerError} wrapping
	 *   the real exception
	 */
	public static HttpError toHttpError(Throwable t) {
		t = unwrap(t);
		if (t instanceof HttpError)
			return (HttpError)t;
		return new InternalServerError(t);
	}
	
	/**
	 * Helper method for OnFail handlers to create an appropriate HTTP error class
	 * for a failed {@link RoutingContext}, by reading the failed status code or
	 * failure exception. 
	 * 
	 * If {@link RoutingContext#failure()} returns a (possibly wrapped) 
	 * {@link HttpError} then that what will be returned, otherwise either an
	 * {@link InternalServerError} will be returned (for an exception failure) or
	 * an appropriate default HTTP status instance according to the failed status code.
	 * 
	 * Note that if the {@link RoutingContext#failed()} {@code == false}, then an {@link OK}
	 * HTTP status class instance will be returned.
	 * @param ctx failed {@code RoutingContext} to investigate
	 * @return an {@code HttpError} instance representing the status of the {@code RoutingContext}
	 */
	public static HttpError toHttpError(RoutingContext ctx) {
		if (!ctx.failed())
			return new OK();
		if (Objects.nonNull(ctx.failure()))
			return toHttpError(ctx.failure());
		if (!HttpStatuses.HTTP_STATUS_CODES.containsKey(ctx.statusCode()))
			return new InternalServerError("Unknown HTTP status code " + ctx.statusCode());
		try {
			return HttpStatuses.create(ctx.statusCode());
		} catch (InstantiationException e) {
			// should never happen, assuming we know all valid HTTP status codes
			return new InternalServerError("Failed to translate failed context to HTTP error");
		}
	}
	
	@Override
	public String toString() {
		return "HTTP " + statusCode + " " + statusText;
	}
}
