/*
 * Decompiled with CFR 0.152.
 */
package tech.checkhim.http;

import com.fasterxml.jackson.core.JsonProcessingException;
import com.fasterxml.jackson.databind.DeserializationFeature;
import com.fasterxml.jackson.databind.ObjectMapper;
import java.io.IOException;
import java.util.concurrent.TimeUnit;
import okhttp3.Cache;
import okhttp3.Interceptor;
import okhttp3.MediaType;
import okhttp3.OkHttpClient;
import okhttp3.Request;
import okhttp3.RequestBody;
import okhttp3.Response;
import okhttp3.ResponseBody;
import tech.checkhim.exceptions.CheckHimException;
import tech.checkhim.exceptions.InvalidFormatException;
import tech.checkhim.exceptions.NetworkRejectedException;
import tech.checkhim.exceptions.PrefixMissingException;
import tech.checkhim.exceptions.SubscriberAbsentException;
import tech.checkhim.exceptions.UnknownSubscriberException;
import tech.checkhim.models.ErrorResponse;

public class HttpClient {
    private static final String USER_AGENT = "CheckHim-Java-SDK/1.0.1";
    private static final MediaType JSON_MEDIA_TYPE = MediaType.get((String)"application/json; charset=utf-8");
    private final OkHttpClient client;
    private final ObjectMapper objectMapper;
    private final String apiKey;
    private final String baseUrl;

    public HttpClient(String apiKey, String baseUrl) {
        this.apiKey = apiKey;
        this.baseUrl = baseUrl;
        this.client = new OkHttpClient.Builder().connectTimeout(30L, TimeUnit.SECONDS).readTimeout(60L, TimeUnit.SECONDS).writeTimeout(60L, TimeUnit.SECONDS).addInterceptor(this::addAuthenticationHeaders).build();
        this.objectMapper = new ObjectMapper().configure(DeserializationFeature.FAIL_ON_UNKNOWN_PROPERTIES, false);
    }

    /*
     * Enabled aggressive block sorting
     * Enabled unnecessary exception pruning
     * Enabled aggressive exception aggregation
     */
    public <T, R> R post(String endpoint, T requestBody, Class<R> responseType) throws CheckHimException {
        try {
            String jsonBody = this.objectMapper.writeValueAsString(requestBody);
            RequestBody body = RequestBody.create((String)jsonBody, (MediaType)JSON_MEDIA_TYPE);
            Request request = new Request.Builder().url(this.baseUrl + endpoint).post(body).build();
            try (Response response = this.client.newCall(request).execute();){
                R r = this.handleResponse(response, responseType);
                return r;
            }
        }
        catch (JsonProcessingException e) {
            throw new CheckHimException("Failed to serialize request body", e);
        }
        catch (IOException e) {
            throw new CheckHimException("Network error occurred while making request", e);
        }
    }

    private <R> R handleResponse(Response response, Class<R> responseType) throws CheckHimException {
        int statusCode = response.code();
        try {
            ResponseBody responseBody = response.body();
            if (responseBody == null) {
                throw new CheckHimException("Empty response body", null, statusCode);
            }
            String responseJson = responseBody.string();
            if (response.isSuccessful()) {
                return (R)this.objectMapper.readValue(responseJson, responseType);
            }
            this.handleErrorResponse(responseJson, statusCode);
            throw new CheckHimException("Unexpected error occurred", null, statusCode);
        }
        catch (JsonProcessingException e) {
            throw new CheckHimException("Failed to parse response JSON", e);
        }
        catch (IOException e) {
            throw new CheckHimException("Failed to read response body", e);
        }
    }

    private void handleErrorResponse(String responseJson, int statusCode) throws CheckHimException {
        try {
            ErrorResponse errorResponse = (ErrorResponse)this.objectMapper.readValue(responseJson, ErrorResponse.class);
            String errorCode = errorResponse.getCode();
            String errorMessage = errorResponse.getDisplayMessage();
            switch (errorCode) {
                case "REJECTED_NETWORK": {
                    throw new NetworkRejectedException(errorMessage, statusCode);
                }
                case "REJECTED_FORMAT": {
                    throw new InvalidFormatException(errorMessage, statusCode);
                }
                case "REJECTED_PREFIX_MISSING": {
                    throw new PrefixMissingException(errorMessage, statusCode);
                }
                case "REJECTED_SUBSCRIBER_ABSENT": {
                    throw new SubscriberAbsentException(errorMessage, statusCode);
                }
                case "REJECTED_UNKNOWN_SUBSCRIBER": {
                    throw new UnknownSubscriberException(errorMessage, statusCode);
                }
            }
            throw new CheckHimException(errorMessage, errorCode, statusCode);
        }
        catch (JsonProcessingException e) {
            throw new CheckHimException("API request failed with status " + statusCode, null, statusCode);
        }
    }

    private Response addAuthenticationHeaders(Interceptor.Chain chain) throws IOException {
        Request originalRequest = chain.request();
        Request authenticatedRequest = originalRequest.newBuilder().addHeader("Authorization", "Bearer " + this.apiKey).addHeader("Content-Type", "application/json").addHeader("User-Agent", USER_AGENT).build();
        return chain.proceed(authenticatedRequest);
    }

    public void close() {
        if (this.client != null) {
            this.client.dispatcher().executorService().shutdown();
            this.client.connectionPool().evictAll();
            Cache cache = this.client.cache();
            if (cache != null) {
                try {
                    cache.close();
                }
                catch (IOException iOException) {
                    // empty catch block
                }
            }
        }
    }
}

