/*
 * Decompiled with CFR 0.152.
 */
package tech.aroma.data.cassandra;

import com.datastax.driver.core.BatchStatement;
import com.datastax.driver.core.ResultSet;
import com.datastax.driver.core.Row;
import com.datastax.driver.core.Session;
import com.datastax.driver.core.Statement;
import com.datastax.driver.core.querybuilder.Delete;
import com.datastax.driver.core.querybuilder.Insert;
import com.datastax.driver.core.querybuilder.QueryBuilder;
import java.util.List;
import java.util.UUID;
import java.util.function.Function;
import javax.inject.Inject;
import org.apache.thrift.TException;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import sir.wellington.alchemy.collections.lists.Lists;
import tech.aroma.data.TokenRepository;
import tech.aroma.data.assertions.AuthenticationAssertions;
import tech.aroma.data.assertions.RequestAssertions;
import tech.aroma.thrift.authentication.AuthenticationToken;
import tech.aroma.thrift.exceptions.InvalidArgumentException;
import tech.aroma.thrift.exceptions.InvalidTokenException;
import tech.aroma.thrift.exceptions.OperationFailedException;
import tech.sirwellington.alchemy.arguments.Arguments;
import tech.sirwellington.alchemy.arguments.assertions.Assertions;
import tech.sirwellington.alchemy.arguments.assertions.StringAssertions;

final class CassandraTokenRepository
implements TokenRepository {
    private static final Logger LOG = LoggerFactory.getLogger(CassandraTokenRepository.class);
    private final Session cassandra;
    private final Function<Row, AuthenticationToken> tokenMapper;

    @Inject
    CassandraTokenRepository(Session cassandra, Function<Row, AuthenticationToken> tokenMapper) {
        Arguments.checkThat((Object)cassandra, (Object[])new Object[]{tokenMapper}).are(Assertions.notNull());
        this.cassandra = cassandra;
        this.tokenMapper = tokenMapper;
    }

    @Override
    public boolean containsToken(String tokenId) throws TException {
        this.checkTokenId(tokenId);
        Statement query = this.createStatementToCheckIfExists(tokenId);
        Row row = this.tryToGetOneRowFrom(query);
        long count = row.getLong(0);
        return count > 0L;
    }

    @Override
    public AuthenticationToken getToken(String tokenId) throws TException, InvalidTokenException {
        this.checkTokenId(tokenId);
        Statement query = this.createQueryToGetToken(tokenId);
        Row row = this.tryToGetOneRowFrom(query);
        AuthenticationToken token = this.tryToConvertRowToToken(row);
        return token;
    }

    @Override
    public void saveToken(AuthenticationToken token) throws TException {
        Arguments.checkThat((Object)token).throwing(InvalidArgumentException.class).is(AuthenticationAssertions.completeToken());
        Statement insertStatement = this.createStatementToInsert(token);
        this.tryToExecute(insertStatement);
        LOG.debug("Token saved in Cassandra");
    }

    @Override
    public List<AuthenticationToken> getTokensBelongingTo(String ownerId) throws TException {
        Arguments.checkThat((Object)ownerId).throwing(InvalidArgumentException.class).usingMessage("ownerId missing").is(StringAssertions.nonEmptyString()).usingMessage("ownerId must be a UUID type").is(StringAssertions.validUUID());
        Statement query = this.createQueryToGetTokensOwnedBy(ownerId);
        ResultSet results = this.tryToGetResultSetFrom(query);
        List tokens = Lists.create();
        for (Row row : results) {
            AuthenticationToken token = this.tryToConvertRowToToken(row);
            tokens.add(token);
        }
        LOG.debug("Found {} tokens owned by {}", (Object)tokens.size(), (Object)ownerId);
        return tokens;
    }

    @Override
    public void deleteToken(String tokenId) throws TException {
        this.checkTokenId(tokenId);
        Statement deleteStatement = this.createStatementToDeleteToken(tokenId);
        this.tryToExecute(deleteStatement);
        LOG.debug("Successfully deleted Token {}", (Object)tokenId);
    }

    private void checkTokenId(String tokenId) throws InvalidArgumentException {
        Arguments.checkThat((Object)tokenId).throwing(InvalidArgumentException.class).usingMessage("missing tokenId").is(StringAssertions.nonEmptyString()).usingMessage("tokenId must be a UUID").is(StringAssertions.validUUID());
    }

    private Statement createStatementToCheckIfExists(String tokenId) {
        UUID tokenUuid = UUID.fromString(tokenId);
        return QueryBuilder.select().countAll().from("Tokens").where(QueryBuilder.eq((String)"token_id", (Object)tokenUuid));
    }

    private Row tryToGetOneRowFrom(Statement query) throws InvalidTokenException, OperationFailedException {
        ResultSet results = this.tryToGetResultSetFrom(query);
        Row row = results.one();
        Arguments.checkThat((Object)row).throwing(InvalidTokenException.class).usingMessage("Token does not exist").is(Assertions.notNull());
        return row;
    }

    private Statement createQueryToGetToken(String tokenId) {
        UUID tokenUuid = UUID.fromString(tokenId);
        return QueryBuilder.select().all().from("Tokens").where(QueryBuilder.eq((String)"token_id", (Object)tokenUuid));
    }

    private ResultSet tryToGetResultSetFrom(Statement statment) throws OperationFailedException {
        ResultSet results;
        try {
            results = this.cassandra.execute(statment);
        }
        catch (Exception ex) {
            LOG.error("Failed to execute Cassandra statement: {}", (Object)statment, (Object)ex);
            throw new OperationFailedException("Could not query for Token:" + ex.getMessage());
        }
        Arguments.checkThat((Object)results).usingMessage("unexpected null result from cassandra").throwing(OperationFailedException.class).is(Assertions.notNull());
        return results;
    }

    private AuthenticationToken tryToConvertRowToToken(Row row) throws OperationFailedException, InvalidTokenException {
        AuthenticationToken token;
        try {
            token = this.tokenMapper.apply(row);
        }
        catch (Exception ex) {
            LOG.error("Could not map Row {} to Token", (Object)row, (Object)ex);
            throw new OperationFailedException("Failed to query for Token: " + ex.getMessage());
        }
        Arguments.checkThat((Object)token).throwing(InvalidTokenException.class).is(AuthenticationAssertions.completeToken());
        return token;
    }

    private Statement createStatementToInsert(AuthenticationToken token) throws InvalidArgumentException {
        UUID tokenId = UUID.fromString(token.tokenId);
        UUID ownerId = UUID.fromString(token.ownerId);
        UUID orgId = null;
        String tokenType = null;
        if (token.tokenType != null) {
            tokenType = token.tokenType.toString();
        }
        String tokenStatus = null;
        if (token.status != null) {
            tokenStatus = token.status.toString();
        }
        if (!RequestAssertions.isNullOrEmpty(token.organizationId)) {
            Arguments.checkThat((Object)token.organizationId).usingMessage("token organizationId must be a UUID type").throwing(InvalidArgumentException.class).is(StringAssertions.validUUID());
            orgId = UUID.fromString(token.organizationId);
        }
        BatchStatement batch = new BatchStatement();
        Insert insertIntoMainTable = QueryBuilder.insertInto((String)"Tokens").value("token_id", (Object)tokenId).value("owner_id", (Object)ownerId).value("organization_id", (Object)orgId).value("owner_name", (Object)token.ownerName).value("time_of_expiration", (Object)token.timeOfExpiration).value("time_of_creation", (Object)token.timeOfCreation).value("token_type", (Object)tokenType).value("token_status", (Object)tokenStatus);
        batch.add((Statement)insertIntoMainTable);
        Insert insertIntoOwnersTable = QueryBuilder.insertInto((String)"Tokens_By_Owner").value("owner_id", (Object)ownerId).value("token_id", (Object)tokenId).value("organization_id", (Object)orgId).value("owner_name", (Object)token.ownerName).value("time_of_expiration", (Object)token.timeOfExpiration).value("time_of_creation", (Object)token.timeOfCreation).value("token_type", (Object)tokenType).value("token_status", (Object)tokenStatus);
        batch.add((Statement)insertIntoOwnersTable);
        return batch;
    }

    private void tryToExecute(Statement statement) throws OperationFailedException {
        try {
            this.cassandra.execute(statement);
        }
        catch (Exception ex) {
            LOG.error("Failed to execute CQL Statement: {}", (Object)statement, (Object)ex);
            throw new OperationFailedException("Could not perform operation: " + ex.getMessage());
        }
    }

    private Statement createQueryToGetTokensOwnedBy(String ownerId) {
        UUID ownerUuid = UUID.fromString(ownerId);
        return QueryBuilder.select().all().from("Tokens_By_Owner").where(QueryBuilder.eq((String)"owner_id", (Object)ownerUuid)).limit(1000);
    }

    private Statement createStatementToDeleteToken(String tokenId) throws TException {
        UUID tokenUuid = UUID.fromString(tokenId);
        AuthenticationToken token = this.getToken(tokenId);
        UUID ownerUuid = UUID.fromString(token.ownerId);
        BatchStatement batch = new BatchStatement();
        Delete.Where deleteFromMainTable = QueryBuilder.delete().all().from("Tokens").where(QueryBuilder.eq((String)"token_id", (Object)tokenUuid));
        batch.add((Statement)deleteFromMainTable);
        Delete.Where deleteFromOwnersTable = QueryBuilder.delete().all().from("Tokens_By_Owner").where(QueryBuilder.eq((String)"owner_id", (Object)ownerUuid));
        batch.add((Statement)deleteFromOwnersTable);
        return batch;
    }
}

