/*
 * Decompiled with CFR 0.152.
 */
package tech.aroma.data.cassandra;

import com.datastax.driver.core.ResultSet;
import com.datastax.driver.core.Row;
import com.datastax.driver.core.Session;
import com.datastax.driver.core.Statement;
import com.datastax.driver.core.querybuilder.QueryBuilder;
import java.util.List;
import java.util.Objects;
import java.util.UUID;
import java.util.function.Function;
import java.util.stream.Collectors;
import javax.inject.Inject;
import org.apache.thrift.TBase;
import org.apache.thrift.TException;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import sir.wellington.alchemy.collections.lists.Lists;
import tech.aroma.data.ReactionRepository;
import tech.aroma.data.assertions.RequestAssertions;
import tech.aroma.thrift.exceptions.InvalidArgumentException;
import tech.aroma.thrift.exceptions.OperationFailedException;
import tech.aroma.thrift.reactions.Reaction;
import tech.sirwellington.alchemy.arguments.Arguments;
import tech.sirwellington.alchemy.arguments.assertions.Assertions;
import tech.sirwellington.alchemy.thrift.ThriftObjects;

final class CassandraReactionRepository
implements ReactionRepository {
    private static final Logger LOG = LoggerFactory.getLogger(CassandraReactionRepository.class);
    private final Session cassandra;
    private final Function<Row, List<Reaction>> reactionMapper;

    @Inject
    CassandraReactionRepository(Session cassandra, Function<Row, List<Reaction>> reactionMapper) {
        Arguments.checkThat((Object)cassandra, (Object[])new Object[]{reactionMapper}).are(Assertions.notNull());
        this.cassandra = cassandra;
        this.reactionMapper = reactionMapper;
    }

    @Override
    public void saveReactionsForUser(String userId, List<Reaction> reactions) throws TException {
        this.checkUserId(userId);
        Statement statement = Lists.isEmpty(reactions) ? this.createQueryToRemoveReactionsFor(userId) : this.createStatementToSaveReactions(userId, reactions);
        this.tryToExecute(statement, "saveReactionsForUser");
    }

    @Override
    public List<Reaction> getReactionsForUser(String userId) throws TException {
        this.checkUserId(userId);
        Statement query = this.createQueryToGetReactionsFor(userId);
        ResultSet results = this.tryToExecute(query, "getReactions");
        if (results.isExhausted()) {
            return Lists.emptyList();
        }
        Row row = results.one();
        if (row == null) {
            return Lists.emptyList();
        }
        return this.reactionMapper.apply(row);
    }

    @Override
    public void saveReactionsForApplication(String appId, List<Reaction> reactions) throws TException {
        this.checkAppId(appId);
        if (Lists.isEmpty(reactions)) {
            Statement statement = this.createQueryToRemoveReactionsFor(appId);
            this.tryToExecute(statement, "deleteReactionsForApplication");
            return;
        }
        Statement statement = this.createStatementToSaveReactions(appId, reactions);
        this.tryToExecute(statement, "saveReactionsForApplication");
    }

    @Override
    public List<Reaction> getReactionsForApplication(String appId) throws TException {
        this.checkAppId(appId);
        Statement query = this.createQueryToGetReactionsFor(appId);
        ResultSet results = this.tryToExecute(query, "getReactionsForApplication");
        Row row = results.one();
        if (row == null) {
            return Lists.emptyList();
        }
        return this.reactionMapper.apply(row);
    }

    private void checkUserId(String userId) throws InvalidArgumentException {
        Arguments.checkThat((Object)userId).throwing(InvalidArgumentException.class).is(RequestAssertions.validUserId());
    }

    private void checkAppId(String appId) throws InvalidArgumentException {
        Arguments.checkThat((Object)appId).throwing(InvalidArgumentException.class).is(RequestAssertions.validApplicationId());
    }

    private ResultSet tryToExecute(Statement statement, String operationName) throws OperationFailedException {
        try {
            return this.cassandra.execute(statement);
        }
        catch (Exception ex) {
            LOG.error("Failed to execute Cassandra Operation: {}", (Object)operationName, (Object)ex);
            throw new OperationFailedException("Could not execute database operation: " + ex.getMessage());
        }
    }

    private Statement createQueryToRemoveReactionsFor(String ownerId) {
        UUID ownerUuid = UUID.fromString(ownerId);
        return QueryBuilder.delete().all().from("Reactions").where(QueryBuilder.eq((String)"owner_id", (Object)ownerUuid));
    }

    private Statement createStatementToSaveReactions(String ownerId, List<Reaction> reactions) {
        UUID ownerUuid = UUID.fromString(ownerId);
        List serializedReactions = reactions.stream().map(this::serialize).filter(Objects::nonNull).collect(Collectors.toList());
        return QueryBuilder.insertInto((String)"Reactions").value("owner_id", (Object)ownerUuid).value("serialized_reactions", serializedReactions);
    }

    private Statement createQueryToGetReactionsFor(String ownerId) {
        UUID ownerUuid = UUID.fromString(ownerId);
        return QueryBuilder.select().all().from("Reactions").where(QueryBuilder.eq((String)"owner_id", (Object)ownerUuid));
    }

    private String serialize(Reaction reaction) {
        try {
            return ThriftObjects.toJson((TBase)reaction);
        }
        catch (TException ex) {
            LOG.warn("Failed to serialize Object {} ", (Object)reaction, (Object)ex);
            return null;
        }
    }
}

