/*
 * Decompiled with CFR 0.152.
 */
package tech.aroma.data.cassandra;

import com.datastax.driver.core.ResultSet;
import com.datastax.driver.core.Row;
import com.datastax.driver.core.Session;
import com.datastax.driver.core.Statement;
import com.datastax.driver.core.querybuilder.Delete;
import com.datastax.driver.core.querybuilder.Ordering;
import com.datastax.driver.core.querybuilder.QueryBuilder;
import java.util.List;
import java.util.UUID;
import java.util.function.Function;
import javax.inject.Inject;
import org.apache.thrift.TBase;
import org.apache.thrift.TException;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import sir.wellington.alchemy.collections.lists.Lists;
import tech.aroma.data.MessageRepository;
import tech.aroma.thrift.LengthOfTime;
import tech.aroma.thrift.Message;
import tech.aroma.thrift.exceptions.InvalidArgumentException;
import tech.aroma.thrift.exceptions.MessageDoesNotExistException;
import tech.aroma.thrift.exceptions.OperationFailedException;
import tech.aroma.thrift.functions.TimeFunctions;
import tech.sirwellington.alchemy.arguments.Arguments;
import tech.sirwellington.alchemy.arguments.assertions.Assertions;
import tech.sirwellington.alchemy.arguments.assertions.StringAssertions;

final class CassandraMessageRepository
implements MessageRepository {
    private static final Logger LOG = LoggerFactory.getLogger(CassandraMessageRepository.class);
    private final Session cassandra;
    private final Function<Row, Message> messageMapper;

    @Inject
    CassandraMessageRepository(Session cassandra, Function<Row, Message> messageMapper) {
        Arguments.checkThat((Object)cassandra, (Object[])new Object[]{messageMapper}).are(Assertions.notNull());
        this.cassandra = cassandra;
        this.messageMapper = messageMapper;
    }

    @Override
    public void saveMessage(Message message, LengthOfTime lifetime) throws TException {
        Arguments.checkThat((Object)message, (Object[])new TBase[]{lifetime}).throwing(InvalidArgumentException.class).is(Assertions.notNull());
        Statement insertStatement = this.createInsertForMessage(message, lifetime);
        Statement updateTotalMessagesByApp = this.createUpdateForMessageByApp(message);
        Statement updateTotalMessageByTitle = this.createUpdateForMessageCounterByTitle(message);
        try {
            this.cassandra.execute(insertStatement);
            LOG.debug("Successfully saved message in Cassandra with a lifetime of {}: {}", (Object)lifetime, (Object)message);
            this.cassandra.executeAsync(updateTotalMessageByTitle);
            this.cassandra.executeAsync(updateTotalMessagesByApp);
            LOG.debug("Successfully Updated Total Message Counters for App {} and title {}", (Object)message.applicationId, (Object)message.title);
        }
        catch (Exception ex) {
            LOG.error("Failed to store message in Cassandra: {}", (Object)message, (Object)ex);
            throw new OperationFailedException("Could save Message");
        }
    }

    @Override
    public Message getMessage(String applicationId, String messageId) throws TException {
        this.checkMessageId(messageId);
        this.checkAppId(applicationId);
        Statement query = this.createQueryForMessageWithId(applicationId, messageId);
        LOG.debug("Querying cassandra for message with ID [{}] and App [{}]", (Object)messageId, (Object)applicationId);
        ResultSet results = this.tryToExecute(query, "Failed to query for message with ID: " + messageId);
        Row row = results.one();
        Arguments.checkThat((Object)row).throwing(MessageDoesNotExistException.class).usingMessage(String.format("No Message with App ID [%s] and Msg ID [%s]", applicationId, messageId)).is(Assertions.notNull());
        Message message = this.createMessageFromRow(row);
        return message;
    }

    @Override
    public void deleteMessage(String applicationId, String messageId) throws TException {
        this.checkAppId(applicationId);
        this.checkMessageId(messageId);
        Statement deleteStatement = this.createDeleteStatementFor(applicationId, messageId);
        this.tryToExecute(deleteStatement, "Failed to delete message with ID: " + messageId);
    }

    @Override
    public boolean containsMessage(String applicationId, String messageId) throws TException {
        this.checkMessageId(messageId);
        this.checkAppId(applicationId);
        Statement query = this.createQueryToCheckIfMessageExists(applicationId, messageId);
        ResultSet results = this.tryToExecute(query, "Could not query for message: " + messageId);
        Row row = results.one();
        this.checkRowIsPresent(row);
        long count = row.getLong(0);
        return count > 0L;
    }

    @Override
    public List<Message> getByHostname(String hostname) throws TException {
        Arguments.checkThat((Object)hostname).usingMessage("missing hostname").throwing(InvalidArgumentException.class).is(StringAssertions.nonEmptyString()).is(StringAssertions.stringWithLengthGreaterThanOrEqualTo((int)1));
        Statement query = this.createQueryToFindMessageByHostname(hostname);
        ResultSet results = this.tryToExecute(query, "Could not query for mesages by hostname: " + hostname);
        List messages = Lists.create();
        for (Row row : results) {
            Message message = this.createMessageFromRow(row);
            messages.add(message);
        }
        LOG.debug("Found {} messages by hostname {}", (Object)messages.size(), (Object)hostname);
        return messages;
    }

    @Override
    public List<Message> getByApplication(String applicationId) throws TException {
        this.checkAppId(applicationId);
        Statement query = this.createQueryToFindMessagesByApplication(applicationId);
        ResultSet results = this.tryToExecute(query, "Could not query for messages by App: " + applicationId);
        List messages = Lists.create();
        for (Row row : results) {
            Message message = this.createMessageFromRow(row);
            messages.add(message);
        }
        LOG.debug("Found {} messages by app with ID [{}]", (Object)messages.size(), (Object)applicationId);
        return messages;
    }

    @Override
    public List<Message> getByTitle(String applicationId, String title) throws TException {
        this.checkAppId(applicationId);
        this.checkTitle(title);
        Statement query = this.createQueryToFindMessagesByTitle(title);
        ResultSet results = this.tryToExecute(query, "Could not get messages by Title: " + title + ", App: " + applicationId);
        List messages = Lists.create();
        for (Row row : results) {
            Message message = this.createMessageFromRow(row);
            messages.add(message);
        }
        LOG.debug("Found {} messages by app with ID [{}]", (Object)messages.size(), (Object)applicationId);
        return messages;
    }

    @Override
    public long getCountByApplication(String applicationId) throws TException {
        this.checkAppId(applicationId);
        Statement query = this.createQueryToCountMessagesByApplication(applicationId);
        ResultSet results = this.tryToExecute(query, "Failed to count messages for App: " + applicationId);
        Row row = results.one();
        this.checkRowIsPresent(row);
        return row.getLong(0);
    }

    @Override
    public void deleteAllMessages(String applicationId) throws TException {
        this.checkAppId(applicationId);
        Statement deleteStatement = this.createStatementToDeleteAllMessagesFor(applicationId);
        this.tryToExecute(deleteStatement, "Failed to delete All Messages for App: " + applicationId);
    }

    private Statement createInsertForMessage(Message message, LengthOfTime lifetime) throws InvalidArgumentException {
        Arguments.checkThat((Object)message.messageId, (Object[])new String[]{message.applicationId}).throwing(InvalidArgumentException.class).are(StringAssertions.nonEmptyString()).are(StringAssertions.validUUID());
        UUID msgId = UUID.fromString(message.messageId);
        UUID appId = UUID.fromString(message.applicationId);
        String urgency = message.urgency != null ? message.urgency.toString() : null;
        Long timeToLive = TimeFunctions.toSeconds((LengthOfTime)lifetime);
        return QueryBuilder.insertInto((String)"Messages").value("message_id", (Object)msgId).value("app_id", (Object)appId).value("app_name", (Object)message.applicationName).value("body", (Object)message.body).value("device_name", (Object)message.deviceName).value("hostname", (Object)message.hostname).value("mac_address", (Object)message.macAddress).value("title", (Object)message.title).value("urgency", (Object)urgency).value("time_created", (Object)message.timeOfCreation).value("time_received", (Object)message.timeMessageReceived).using(QueryBuilder.ttl((int)timeToLive.intValue()));
    }

    private Statement createUpdateForMessageByApp(Message message) {
        UUID appId = UUID.fromString(message.applicationId);
        return QueryBuilder.update((String)"Messages_Totals_By_App").where(QueryBuilder.eq((String)"app_id", (Object)appId)).with(QueryBuilder.incr((String)"total_messages", (long)1L));
    }

    private Statement createUpdateForMessageCounterByTitle(Message message) {
        UUID appId = UUID.fromString(message.applicationId);
        return QueryBuilder.update((String)"Messages_Totals_By_Title").where(QueryBuilder.eq((String)"app_id", (Object)appId)).and(QueryBuilder.eq((String)"title", (Object)message.title)).with(QueryBuilder.incr((String)"total_messages", (long)1L));
    }

    private void checkMessageId(String messageId) throws InvalidArgumentException {
        Arguments.checkThat((Object)messageId).usingMessage("missing messageId").throwing(InvalidArgumentException.class).is(StringAssertions.nonEmptyString()).usingMessage("messageId must be a UUID").is(StringAssertions.validUUID());
    }

    private Statement createQueryForMessageWithId(String applicationId, String messageId) {
        UUID appId = UUID.fromString(applicationId);
        UUID msgId = UUID.fromString(messageId);
        return QueryBuilder.select().all().from("Messages").where(QueryBuilder.eq((String)"message_id", (Object)msgId)).and(QueryBuilder.eq((String)"app_id", (Object)appId)).limit(2);
    }

    private void checkRowIsPresent(Row row) throws OperationFailedException {
        Arguments.checkThat((Object)row).usingMessage("query produced no rows").throwing(OperationFailedException.class).is(Assertions.notNull());
    }

    private Message createMessageFromRow(Row row) throws OperationFailedException {
        return this.messageMapper.apply(row);
    }

    private Statement createDeleteStatementFor(String applicationId, String messageId) {
        UUID msgId = UUID.fromString(messageId);
        UUID appId = UUID.fromString(applicationId);
        Delete.Where deleteFromMainTable = QueryBuilder.delete().all().from("Messages").where(QueryBuilder.eq((String)"app_id", (Object)appId)).and(QueryBuilder.eq((String)"message_id", (Object)msgId));
        return deleteFromMainTable;
    }

    private Statement createQueryToCheckIfMessageExists(String applicationId, String messageId) {
        UUID msgId = UUID.fromString(messageId);
        UUID appId = UUID.fromString(applicationId);
        return QueryBuilder.select().countAll().from("Messages").where(QueryBuilder.eq((String)"app_id", (Object)appId)).and(QueryBuilder.eq((String)"message_id", (Object)msgId));
    }

    private Statement createQueryToFindMessageByHostname(String hostname) {
        return QueryBuilder.select().all().from("Messages").where(QueryBuilder.eq((String)"hostname", (Object)hostname));
    }

    private void checkTitle(String title) throws InvalidArgumentException {
        Arguments.checkThat((Object)title).throwing(InvalidArgumentException.class).is(StringAssertions.nonEmptyString()).is(StringAssertions.stringWithLengthGreaterThanOrEqualTo((int)2));
    }

    private Statement createQueryToFindMessagesByApplication(String applicationId) {
        UUID appId = UUID.fromString(applicationId);
        return QueryBuilder.select().all().from("Messages").where(QueryBuilder.eq((String)"app_id", (Object)appId)).orderBy(new Ordering[]{QueryBuilder.desc((String)"message_id")}).limit(3000);
    }

    private Statement createQueryToCountMessagesByApplication(String applicationId) {
        UUID appId = UUID.fromString(applicationId);
        return QueryBuilder.select().countAll().from("Messages").where(QueryBuilder.eq((String)"app_id", (Object)appId));
    }

    private Statement createQueryToFindMessagesByTitle(String title) {
        return QueryBuilder.select().all().from("Messages").where(QueryBuilder.eq((String)"title", (Object)title));
    }

    private void checkAppId(String applicationId) throws InvalidArgumentException {
        Arguments.checkThat((Object)applicationId).throwing(InvalidArgumentException.class).usingMessage("missing appId").is(StringAssertions.nonEmptyString()).usingMessage("appId must be a UUID Type").is(StringAssertions.validUUID());
    }

    private Statement createStatementToDeleteAllMessagesFor(String applicationId) {
        UUID appId = UUID.fromString(applicationId);
        Delete.Where deleteFromMainTable = QueryBuilder.delete().all().from("Messages").where(QueryBuilder.eq((String)"app_id", (Object)appId));
        return deleteFromMainTable;
    }

    private ResultSet tryToExecute(Statement statement, String errorMessage) throws OperationFailedException {
        ResultSet results;
        try {
            results = this.cassandra.execute(statement);
        }
        catch (Exception ex) {
            LOG.error("Failed to execute Cassandra statement: {}", (Object)statement, (Object)ex);
            throw new OperationFailedException(errorMessage + " | " + ex.getMessage());
        }
        Arguments.checkThat((Object)results).throwing(OperationFailedException.class).usingMessage("Cassandra returned null response").is(Assertions.notNull());
        return results;
    }
}

