/*
 * Decompiled with CFR 0.152.
 */
package tech.aroma.data.cassandra;

import com.datastax.driver.core.ResultSet;
import com.datastax.driver.core.Row;
import com.datastax.driver.core.Session;
import com.datastax.driver.core.Statement;
import com.datastax.driver.core.querybuilder.QueryBuilder;
import java.time.Instant;
import java.util.UUID;
import java.util.function.Function;
import javax.inject.Inject;
import org.apache.thrift.TException;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import tech.aroma.data.MediaRepository;
import tech.aroma.thrift.Dimension;
import tech.aroma.thrift.Image;
import tech.aroma.thrift.exceptions.DoesNotExistException;
import tech.aroma.thrift.exceptions.InvalidArgumentException;
import tech.aroma.thrift.exceptions.OperationFailedException;
import tech.sirwellington.alchemy.arguments.Arguments;
import tech.sirwellington.alchemy.arguments.assertions.Assertions;
import tech.sirwellington.alchemy.arguments.assertions.NumberAssertions;
import tech.sirwellington.alchemy.arguments.assertions.StringAssertions;

final class CassandraMediaRepository
implements MediaRepository {
    private static final Logger LOG = LoggerFactory.getLogger(CassandraMediaRepository.class);
    private final Session cassandra;
    private final Function<Row, Image> imageMapper;

    @Inject
    CassandraMediaRepository(Session cassandra, Function<Row, Image> imageMapper) {
        Arguments.checkThat((Object)cassandra, (Object[])new Object[]{imageMapper}).are(Assertions.notNull());
        this.cassandra = cassandra;
        this.imageMapper = imageMapper;
    }

    @Override
    public void saveMedia(String mediaId, Image image) throws TException {
        this.checkMediaId(mediaId);
        this.checkImage(image);
        Statement insertStatement = this.createStatementToSaveImage(mediaId, image);
        this.tryToExecute(insertStatement, "Could not save Media in Cassandra: " + mediaId);
        LOG.debug("Successfully saved media with ID {} in Cassandra", (Object)mediaId);
    }

    @Override
    public Image getMedia(String mediaId) throws DoesNotExistException, TException {
        this.checkMediaId(mediaId);
        Statement query = this.createQueryToGetImage(mediaId);
        ResultSet results = this.tryToExecute(query, "Could not query Cassandra for Media with ID:" + mediaId);
        this.checkResultsNotMissing(results);
        Row row = results.one();
        this.ensureRowExists(row, mediaId);
        Image image = this.converRowToImage(row);
        return image;
    }

    @Override
    public void deleteMedia(String mediaId) throws TException {
        this.checkMediaId(mediaId);
        Statement deleteStatement = this.createStatementToDeleteMedia(mediaId);
        this.tryToExecute(deleteStatement, "Failed to delete media from Cassandra: " + mediaId);
    }

    @Override
    public void saveThumbnail(String mediaId, Dimension dimension, Image thumbnail) throws TException {
        this.checkMediaId(mediaId);
        this.checkImage(thumbnail);
        this.checkDimension(dimension);
        Statement insertStatement = this.createStatementToSaveThumbnail(mediaId, dimension, thumbnail);
        this.tryToExecute(insertStatement, "Failed to save Thumbnail in Cassandra. Media ID: " + mediaId + " Dimension: " + dimension);
    }

    @Override
    public Image getThumbnail(String mediaId, Dimension dimension) throws DoesNotExistException, TException {
        this.checkMediaId(mediaId);
        this.checkDimension(dimension);
        Statement query = this.createQueryToGetThumbnail(mediaId, dimension);
        ResultSet results = this.tryToExecute(query, "Failed to Query Cassandra for Thumbnail with Media ID: " + mediaId + " Dimension: " + dimension);
        this.checkResultsNotMissing(results);
        Row row = results.one();
        this.ensureRowExists(row, mediaId);
        Image thumbnail = this.converRowToImage(row);
        return thumbnail;
    }

    @Override
    public void deleteThumbnail(String mediaId, Dimension dimension) throws TException {
        this.checkMediaId(mediaId);
        this.checkDimension(dimension);
        Statement deleteStatement = this.createStatementToDeleteThubmnail(mediaId, dimension);
        this.tryToExecute(deleteStatement, "Failed to delete Thumbnail from Cassandra: " + mediaId + " Dimension: " + dimension);
    }

    @Override
    public void deleteAllThumbnails(String mediaId) throws TException {
        this.checkMediaId(mediaId);
        Statement deleteStatement = this.createStatementToDeleteAllThumbnailsFor(mediaId);
        this.tryToExecute(deleteStatement, "Failed to delete all Thumbnails belonging to Media: " + mediaId);
    }

    private void checkMediaId(String mediaId) throws InvalidArgumentException {
        Arguments.checkThat((Object)mediaId).throwing(InvalidArgumentException.class).usingMessage("mediaId missing").is(StringAssertions.nonEmptyString()).usingMessage("mediaId must be a UUID").is(StringAssertions.validUUID());
    }

    private void checkImage(Image image) throws InvalidArgumentException {
        Arguments.checkThat((Object)image).throwing(InvalidArgumentException.class).usingMessage("image cannot be null").is(Assertions.notNull());
        byte[] data = image.getData();
        Arguments.checkThat((Object)data).throwing(InvalidArgumentException.class).usingMessage("Image is missing data").is(Assertions.notNull());
        Arguments.checkThat((Object)data.length).throwing(InvalidArgumentException.class).usingMessage("Image data is empty").is(NumberAssertions.greaterThan((int)0));
    }

    private ResultSet tryToExecute(Statement statement, String errorMessage) throws OperationFailedException {
        try {
            return this.cassandra.execute(statement);
        }
        catch (Exception ex) {
            LOG.error("Failed to execute Cassandra Statement: {}", (Object)errorMessage, (Object)ex);
            throw new OperationFailedException(errorMessage + " | " + ex.getMessage());
        }
    }

    private Statement createStatementToSaveImage(String mediaId, Image image) {
        UUID mediaUuid = UUID.fromString(mediaId);
        String type = image.imageType != null ? image.imageType.toString() : null;
        Dimension dimension = image.dimension != null ? image.dimension : new Dimension();
        return QueryBuilder.insertInto((String)"Media").value("media_id", (Object)mediaUuid).value("media_type", (Object)type).value("creation_time", (Object)Instant.now().toEpochMilli()).value("binary", (Object)image.data).value("width", (Object)dimension.width).value("height", (Object)dimension.height);
    }

    private Statement createQueryToGetImage(String mediaId) {
        UUID mediaUuid = UUID.fromString(mediaId);
        return QueryBuilder.select().all().from("Media").where(QueryBuilder.eq((String)"media_id", (Object)mediaUuid));
    }

    private void checkResultsNotMissing(ResultSet results) throws OperationFailedException {
        Arguments.checkThat((Object)results).throwing(OperationFailedException.class).usingMessage("Cassandra returned null ResultSet").is(Assertions.notNull());
    }

    private void ensureRowExists(Row row, String mediaId) throws DoesNotExistException {
        Arguments.checkThat((Object)row).throwing(DoesNotExistException.class).usingMessage("Media does not exist: " + mediaId).is(Assertions.notNull());
    }

    private Image converRowToImage(Row row) {
        return this.imageMapper.apply(row);
    }

    private Statement createStatementToDeleteMedia(String mediaId) {
        UUID mediaUuid = UUID.fromString(mediaId);
        return QueryBuilder.delete().all().from("Media").where(QueryBuilder.eq((String)"media_id", (Object)mediaUuid));
    }

    private void checkDimension(Dimension dimension) throws InvalidArgumentException {
        Arguments.checkThat((Object)dimension).usingMessage("dimension cannot be null").throwing(InvalidArgumentException.class).is(Assertions.notNull());
        Arguments.checkThat((Object)dimension.width).usingMessage("dimension width must be >0").throwing(InvalidArgumentException.class).is(NumberAssertions.greaterThan((int)0));
        Arguments.checkThat((Object)dimension.height).usingMessage("dimension height must be >0").throwing(InvalidArgumentException.class).is(NumberAssertions.greaterThan((int)0));
    }

    private Statement createStatementToSaveThumbnail(String mediaId, Dimension dimension, Image thumbnail) {
        UUID mediaUuid = UUID.fromString(mediaId);
        String dimensionString = dimension.toString();
        String mediaType = thumbnail.imageType != null ? thumbnail.imageType.toString() : null;
        return QueryBuilder.insertInto((String)"Media_Thumbnails").value("media_id", (Object)mediaUuid).value("dimension", (Object)dimensionString).value("width", (Object)dimension.width).value("height", (Object)dimension.height).value("creation_time", (Object)Instant.now().toEpochMilli()).value("binary", (Object)thumbnail.data).value("media_type", (Object)mediaType);
    }

    private Statement createQueryToGetThumbnail(String mediaId, Dimension dimension) {
        UUID mediaUuid = UUID.fromString(mediaId);
        String dimensionString = dimension.toString();
        return QueryBuilder.select().all().from("Media_Thumbnails").where(QueryBuilder.eq((String)"media_id", (Object)mediaUuid)).and(QueryBuilder.eq((String)"dimension", (Object)dimensionString));
    }

    private Statement createStatementToDeleteThubmnail(String mediaId, Dimension dimension) {
        UUID mediaUuid = UUID.fromString(mediaId);
        String dimensionString = dimension.toString();
        return QueryBuilder.delete().all().from("Media_Thumbnails").where(QueryBuilder.eq((String)"media_id", (Object)mediaUuid)).and(QueryBuilder.eq((String)"dimension", (Object)dimensionString));
    }

    private Statement createStatementToDeleteAllThumbnailsFor(String mediaId) {
        UUID mediaUuid = UUID.fromString(mediaId);
        return QueryBuilder.delete().all().from("Media_Thumbnails").where(QueryBuilder.eq((String)"media_id", (Object)mediaUuid));
    }
}

