/*
 * Decompiled with CFR 0.152.
 */
package tech.aroma.data.cassandra;

import com.datastax.driver.core.BatchStatement;
import com.datastax.driver.core.ResultSet;
import com.datastax.driver.core.Row;
import com.datastax.driver.core.Session;
import com.datastax.driver.core.Statement;
import com.datastax.driver.core.querybuilder.Delete;
import com.datastax.driver.core.querybuilder.Insert;
import com.datastax.driver.core.querybuilder.QueryBuilder;
import java.time.Instant;
import java.util.List;
import java.util.UUID;
import java.util.function.Function;
import javax.inject.Inject;
import org.apache.thrift.TException;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import sir.wellington.alchemy.collections.lists.Lists;
import tech.aroma.data.FollowerRepository;
import tech.aroma.data.assertions.RequestAssertions;
import tech.aroma.thrift.Application;
import tech.aroma.thrift.User;
import tech.aroma.thrift.exceptions.InvalidArgumentException;
import tech.aroma.thrift.exceptions.OperationFailedException;
import tech.sirwellington.alchemy.arguments.Arguments;
import tech.sirwellington.alchemy.arguments.assertions.Assertions;

final class CassandraFollowerRepository
implements FollowerRepository {
    private static final Logger LOG = LoggerFactory.getLogger(CassandraFollowerRepository.class);
    private final Session cassandra;
    private final Function<Row, User> userMapper;
    private final Function<Row, Application> applicationMapper;

    @Inject
    CassandraFollowerRepository(Session cassandra, Function<Row, User> userMapper, Function<Row, Application> applicationMapper) {
        Arguments.checkThat((Object)cassandra, (Object[])new Object[]{userMapper, applicationMapper}).are(Assertions.notNull());
        this.cassandra = cassandra;
        this.applicationMapper = applicationMapper;
        this.userMapper = userMapper;
    }

    @Override
    public void saveFollowing(User user, Application application) throws TException {
        Arguments.checkThat((Object)user).throwing(InvalidArgumentException.class).is(RequestAssertions.validUser());
        Arguments.checkThat((Object)application).throwing(InvalidArgumentException.class).is(RequestAssertions.validApplication());
        Statement insertStatement = this.createStatementToSaveFollowing(user, application);
        try {
            this.cassandra.execute(insertStatement);
        }
        catch (Exception ex) {
            LOG.error("Failed to save following: User: [{}] App: [{}]", new Object[]{user.userId, application.applicationId, ex});
            throw new OperationFailedException("Could not save following in Cassandra: " + ex.getMessage());
        }
    }

    @Override
    public void deleteFollowing(String userId, String applicationId) throws TException {
        this.checkUserId(userId);
        this.checkAppId(applicationId);
        Statement deleteStatement = this.createDeleteStatementFor(userId, applicationId);
        try {
            this.cassandra.execute(deleteStatement);
        }
        catch (Exception ex) {
            LOG.error("Failed to delete the following between User: [{}] App: [{}]", new Object[]{userId, applicationId, ex});
            throw new OperationFailedException("Could not delete following: " + ex.getMessage());
        }
    }

    @Override
    public boolean followingExists(String userId, String applicationId) throws TException {
        ResultSet results;
        this.checkUserId(userId);
        this.checkAppId(applicationId);
        Statement query = this.createStatementToCheckIfFollowingExists(userId, applicationId);
        try {
            results = this.cassandra.execute(query);
        }
        catch (Exception ex) {
            LOG.error("Failed to query for following between User: [{}] App: [{}]", new Object[]{userId, applicationId, ex});
            throw new OperationFailedException("Could not query for following: " + ex.getMessage());
        }
        Row row = results.one();
        this.checkRowExists(row);
        long count = row.getLong(0);
        return count > 0L;
    }

    @Override
    public List<Application> getApplicationsFollowedBy(String userId) throws TException {
        ResultSet results;
        this.checkUserId(userId);
        Statement query = this.createQueryForAppsFollowedBy(userId);
        try {
            results = this.cassandra.execute(query);
        }
        catch (Exception ex) {
            LOG.error("Failed to query Cassandra for apps followed by User: [{}]", (Object)userId, (Object)ex);
            throw new OperationFailedException("Could not find apps followed by user: " + ex.getMessage());
        }
        List apps = Lists.create();
        for (Row row : results) {
            Application app = this.createAppFromRow(row);
            apps.add(app);
        }
        LOG.debug("Found {} apps followed by {}", (Object)apps.size(), (Object)userId);
        return apps;
    }

    @Override
    public List<User> getApplicationFollowers(String applicationId) throws TException {
        ResultSet results;
        this.checkAppId(applicationId);
        Statement query = this.createQueryForFollowersOfApp(applicationId);
        try {
            results = this.cassandra.execute(query);
        }
        catch (Exception ex) {
            LOG.error("Failed to query for App's followed: App: [{}]", (Object)applicationId, (Object)ex);
            throw new OperationFailedException("Could not query for App's Followers: " + ex.getMessage());
        }
        List followers = Lists.create();
        for (Row row : results) {
            User follower = this.createUserFromRow(row);
            followers.add(follower);
        }
        LOG.debug("Found {} Users followed App [{}]", (Object)followers.size(), (Object)applicationId);
        return followers;
    }

    private Statement createStatementToSaveFollowing(User user, Application app) {
        UUID userId = UUID.fromString(user.userId);
        UUID appId = UUID.fromString(app.applicationId);
        BatchStatement batch = new BatchStatement();
        Insert insertIntoAppFollowersTable = QueryBuilder.insertInto((String)"Follow_Application_Followers").value("app_id", (Object)appId).value("user_id", (Object)userId).value("app_name", (Object)app.name).value("user_first_name", (Object)user.firstName).value("time_of_follow", (Object)Instant.now().toEpochMilli());
        batch.add((Statement)insertIntoAppFollowersTable);
        Insert insertIntoUserFollowingsTable = QueryBuilder.insertInto((String)"Follow_User_Followings").value("app_id", (Object)appId).value("user_id", (Object)userId).value("app_name", (Object)app.name).value("user_first_name", (Object)user.firstName).value("time_of_follow", (Object)Instant.now().toEpochMilli());
        batch.add((Statement)insertIntoUserFollowingsTable);
        return batch;
    }

    private Statement createDeleteStatementFor(String userId, String applicationId) {
        UUID appUuid = UUID.fromString(applicationId);
        UUID userUuid = UUID.fromString(userId);
        BatchStatement batch = new BatchStatement();
        Delete.Where deleteFromAppFollowersTable = QueryBuilder.delete().all().from("Follow_Application_Followers").where(QueryBuilder.eq((String)"app_id", (Object)appUuid)).and(QueryBuilder.eq((String)"user_id", (Object)userUuid));
        batch.add((Statement)deleteFromAppFollowersTable);
        Delete.Where deleteFromUserFollowingsTable = QueryBuilder.delete().all().from("Follow_User_Followings").where(QueryBuilder.eq((String)"app_id", (Object)appUuid)).and(QueryBuilder.eq((String)"user_id", (Object)userUuid));
        batch.add((Statement)deleteFromUserFollowingsTable);
        return batch;
    }

    private Statement createStatementToCheckIfFollowingExists(String userId, String applicationId) {
        UUID appUuid = UUID.fromString(applicationId);
        UUID userUuid = UUID.fromString(userId);
        return QueryBuilder.select().countAll().from("Follow_Application_Followers").where(QueryBuilder.eq((String)"user_id", (Object)userUuid)).and(QueryBuilder.eq((String)"app_id", (Object)appUuid));
    }

    private void checkRowExists(Row row) throws OperationFailedException {
        Arguments.checkThat((Object)row).throwing(OperationFailedException.class).usingMessage("Query Failed").is(Assertions.notNull());
    }

    private Statement createQueryForAppsFollowedBy(String userId) {
        UUID userUuid = UUID.fromString(userId);
        return QueryBuilder.select().column("app_name").as("name").column("app_id").as("app_id").column("user_first_name").as("first_name").column("time_of_follow").column("user_id").from("Follow_User_Followings").where(QueryBuilder.eq((String)"user_id", (Object)userUuid));
    }

    private Statement createQueryForFollowersOfApp(String applicationId) {
        UUID appUuid = UUID.fromString(applicationId);
        return QueryBuilder.select().column("app_name").as("name").column("app_id").as("app_id").column("user_first_name").as("first_name").column("time_of_follow").column("user_id").from("Follow_Application_Followers").where(QueryBuilder.eq((String)"app_id", (Object)appUuid));
    }

    private Application createAppFromRow(Row row) {
        return this.applicationMapper.apply(row);
    }

    private User createUserFromRow(Row row) {
        return this.userMapper.apply(row);
    }

    private void checkUserId(String userId) throws InvalidArgumentException {
        Arguments.checkThat((Object)userId).throwing(ex -> new InvalidArgumentException(ex.getMessage())).is(RequestAssertions.validUserId());
    }

    private void checkAppId(String applicationId) throws InvalidArgumentException {
        Arguments.checkThat((Object)applicationId).throwing(ex -> new InvalidArgumentException(ex.getMessage())).is(RequestAssertions.validApplicationId());
    }
}

