/*
 * Decompiled with CFR 0.152.
 */
package tech.aroma.data.cassandra;

import com.datastax.driver.core.ResultSet;
import com.datastax.driver.core.Row;
import com.datastax.driver.core.Session;
import com.datastax.driver.core.Statement;
import com.datastax.driver.core.querybuilder.QueryBuilder;
import java.util.Date;
import java.util.UUID;
import javax.inject.Inject;
import org.apache.thrift.TException;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import tech.aroma.data.CredentialRepository;
import tech.aroma.data.assertions.RequestAssertions;
import tech.aroma.thrift.exceptions.DoesNotExistException;
import tech.aroma.thrift.exceptions.InvalidArgumentException;
import tech.aroma.thrift.exceptions.OperationFailedException;
import tech.sirwellington.alchemy.annotations.access.Internal;
import tech.sirwellington.alchemy.arguments.Arguments;
import tech.sirwellington.alchemy.arguments.assertions.Assertions;
import tech.sirwellington.alchemy.arguments.assertions.StringAssertions;

@Internal
final class CassandraCredentialsRepository
implements CredentialRepository {
    private static final Logger LOG = LoggerFactory.getLogger(CassandraCredentialsRepository.class);
    private final Session cassandra;

    @Inject
    CassandraCredentialsRepository(Session cassandra) {
        Arguments.checkThat((Object)cassandra).is(Assertions.notNull());
        this.cassandra = cassandra;
    }

    @Override
    public void saveEncryptedPassword(String userId, String encryptedPassword) throws TException {
        this.checkUserId(userId);
        this.checkPassword(encryptedPassword);
        Statement insertStatement = this.createInsertStatementFor(userId, encryptedPassword);
        this.tryToExecute(insertStatement, "Could not insert credentials for: " + userId);
        LOG.debug("Successfully stored credentials for {} in Cassandra", (Object)userId);
    }

    @Override
    public boolean containsEncryptedPassword(String userId) throws TException {
        this.checkUserId(userId);
        Statement query = this.createQueryToCheckIfExists(userId);
        ResultSet results = this.tryToExecute(query, "Could not query if credentials exist for: " + userId);
        this.checkResultExist(results);
        Row row = results.one();
        this.ensureRowNotEmpty(row);
        long count = row.getLong(0);
        return count > 0L;
    }

    @Override
    public String getEncryptedPassword(String userId) throws TException {
        this.checkUserId(userId);
        Statement query = this.createQueryToGetEncryptedPasswordFor(userId);
        ResultSet results = this.tryToExecute(query, "Could not query for Credentials of " + userId);
        this.checkResultExist(results);
        Row row = results.one();
        this.ensureRowNotEmpty(row);
        String encryptedPassword = row.getString("encrypted_password");
        this.ensureNotEmpty(encryptedPassword);
        return encryptedPassword;
    }

    @Override
    public void deleteEncryptedPassword(String userId) throws TException {
        this.checkUserId(userId);
        Statement deleteStatement = this.createStatementToDeleteCredentialsFor(userId);
        this.tryToExecute(deleteStatement, "Could not delete credentials for: " + userId);
        LOG.debug("Successfully deleted credentials for user [{}]", (Object)userId);
    }

    private void checkUserId(String userId) throws InvalidArgumentException {
        Arguments.checkThat((Object)userId).throwing(InvalidArgumentException.class).is(RequestAssertions.validUserId());
    }

    private void checkPassword(String encryptedPassword) throws InvalidArgumentException {
        Arguments.checkThat((Object)encryptedPassword).throwing(InvalidArgumentException.class).is(StringAssertions.nonEmptyString()).is(StringAssertions.stringWithLengthGreaterThan((int)1));
    }

    private ResultSet tryToExecute(Statement statement, String errorMessage) throws OperationFailedException {
        try {
            return this.cassandra.execute(statement);
        }
        catch (Exception ex) {
            LOG.error(errorMessage, (Throwable)ex);
            throw new OperationFailedException(errorMessage + ": " + ex.getMessage());
        }
    }

    private Statement createInsertStatementFor(String userId, String encryptedPassword) {
        UUID userUuid = UUID.fromString(userId);
        return QueryBuilder.insertInto((String)"Credentials").value("user_id", (Object)userUuid).value("time_created", (Object)new Date()).value("encrypted_password", (Object)encryptedPassword);
    }

    private Statement createQueryToCheckIfExists(String userId) {
        UUID userUuid = UUID.fromString(userId);
        return QueryBuilder.select().countAll().from("Credentials").where(QueryBuilder.eq((String)"user_id", (Object)userUuid));
    }

    private void checkResultExist(ResultSet results) throws OperationFailedException {
        Arguments.checkThat((Object)results).throwing(OperationFailedException.class).is(Assertions.notNull());
    }

    private void ensureRowNotEmpty(Row row) throws DoesNotExistException {
        Arguments.checkThat((Object)row).throwing(DoesNotExistException.class).is(Assertions.notNull());
    }

    private Statement createQueryToGetEncryptedPasswordFor(String userId) {
        UUID userUuid = UUID.fromString(userId);
        return QueryBuilder.select((String[])new String[]{"encrypted_password"}).from("Credentials").where(QueryBuilder.eq((String)"user_id", (Object)userUuid));
    }

    private void ensureNotEmpty(String encryptedPassword) throws DoesNotExistException {
        Arguments.checkThat((Object)encryptedPassword).throwing(DoesNotExistException.class).usingMessage("Password is missing or invalid").is(StringAssertions.nonEmptyString());
    }

    private Statement createStatementToDeleteCredentialsFor(String userId) {
        UUID userUuid = UUID.fromString(userId);
        return QueryBuilder.delete().all().from("Credentials").where(QueryBuilder.eq((String)"user_id", (Object)userUuid));
    }
}

