/*
 * Decompiled with CFR 0.152.
 */
package tech.aroma.data.cassandra;

import com.datastax.driver.core.BatchStatement;
import com.datastax.driver.core.ResultSet;
import com.datastax.driver.core.Row;
import com.datastax.driver.core.Session;
import com.datastax.driver.core.Statement;
import com.datastax.driver.core.querybuilder.Delete;
import com.datastax.driver.core.querybuilder.Insert;
import com.datastax.driver.core.querybuilder.QueryBuilder;
import java.time.Duration;
import java.util.List;
import java.util.Set;
import java.util.UUID;
import java.util.function.Function;
import java.util.stream.Collectors;
import javax.inject.Inject;
import org.apache.thrift.TException;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import sir.wellington.alchemy.collections.lists.Lists;
import sir.wellington.alchemy.collections.sets.Sets;
import tech.aroma.data.ApplicationRepository;
import tech.aroma.data.assertions.RequestAssertions;
import tech.aroma.thrift.Application;
import tech.aroma.thrift.exceptions.ApplicationDoesNotExistException;
import tech.aroma.thrift.exceptions.InvalidArgumentException;
import tech.aroma.thrift.exceptions.OperationFailedException;
import tech.sirwellington.alchemy.arguments.Arguments;
import tech.sirwellington.alchemy.arguments.assertions.Assertions;
import tech.sirwellington.alchemy.arguments.assertions.StringAssertions;

final class CassandraApplicationRepository
implements ApplicationRepository {
    private static final Logger LOG = LoggerFactory.getLogger(CassandraApplicationRepository.class);
    private static final Duration DEFAULT_RECENT_DURATION = Duration.ofDays(5L);
    private final Session cassandra;
    private final Function<Row, Application> applicationMapper;

    @Inject
    CassandraApplicationRepository(Session cassandra, Function<Row, Application> applicationMapper) {
        Arguments.checkThat((Object)cassandra, (Object[])new Object[]{applicationMapper}).are(Assertions.notNull());
        this.cassandra = cassandra;
        this.applicationMapper = applicationMapper;
    }

    @Override
    public void saveApplication(Application application) throws TException {
        Arguments.checkThat((Object)application).throwing(InvalidArgumentException.class).is(RequestAssertions.validApplication());
        Statement statement = this.createStatementToSave(application);
        try {
            this.cassandra.execute(statement);
            LOG.debug("Successfully saved Application in Cassandra: {}", (Object)application);
        }
        catch (Exception ex) {
            LOG.error("Failed to store Application in Cassandra: {}", (Object)application, (Object)ex);
            throw new OperationFailedException("Could not save Application: " + ex.getMessage());
        }
    }

    @Override
    public void deleteApplication(String applicationId) throws TException {
        this.checkApplicationId(applicationId);
        Application app = this.getById(applicationId);
        Statement statement = this.createDeleteStatementFor(app);
        try {
            this.cassandra.execute(statement);
            LOG.debug("Successfully deleted Application with ID {}", (Object)applicationId);
        }
        catch (Exception ex) {
            LOG.error("Failed to delete application with ID [{}] from Cassandra", (Object)applicationId, (Object)ex);
            throw new OperationFailedException("Could not delete Application with ID: " + applicationId);
        }
    }

    @Override
    public Application getById(String applicationId) throws TException {
        ResultSet results;
        this.checkApplicationId(applicationId);
        Statement query = this.createQueryForAppWithId(applicationId);
        try {
            results = this.cassandra.execute(query);
        }
        catch (Exception ex) {
            LOG.error("Failed to query for application with ID {}", (Object)applicationId, (Object)ex);
            throw new OperationFailedException("Could not Query Application with ID: " + applicationId);
        }
        Row row = results.one();
        this.checkRowNotMissing(applicationId, row);
        Application app = this.createApplicationFromRow(row);
        return app;
    }

    @Override
    public boolean containsApplication(String applicationId) throws TException {
        ResultSet results;
        this.checkApplicationId(applicationId);
        Statement query = this.createQueryToCheckIfAppIdExists(applicationId);
        try {
            results = this.cassandra.execute(query);
        }
        catch (Exception ex) {
            LOG.error("Failed to check Application existence for [{}]", (Object)applicationId, (Object)ex);
            throw new OperationFailedException("Could not check for application existence: " + applicationId);
        }
        Row row = results.one();
        this.checkRowNotMissing(applicationId, row);
        long count = row.getLong(0);
        return count > 0L;
    }

    @Override
    public List<Application> getApplicationsOwnedBy(String userId) throws TException {
        ResultSet results;
        Arguments.checkThat((Object)userId).throwing(InvalidArgumentException.class).is(RequestAssertions.validUserId());
        Statement query = this.createQueryForAppsOwnedBy(userId);
        try {
            results = this.cassandra.execute(query);
        }
        catch (Exception ex) {
            LOG.error("Failed to query for Apps owned by {}", (Object)userId, (Object)ex);
            throw new OperationFailedException("Could not determine Apps owned by user: " + userId);
        }
        List apps = Lists.create();
        for (Row row : results) {
            if (row == null) continue;
            Application app = this.createApplicationFromRow(row);
            apps.add(app);
        }
        LOG.debug("Found {} apps owned by user {}", (Object)apps.size(), (Object)userId);
        return apps;
    }

    @Override
    public List<Application> getApplicationsByOrg(String orgId) throws TException {
        ResultSet results;
        Arguments.checkThat((Object)orgId).throwing(InvalidArgumentException.class).is(RequestAssertions.validOrgId());
        Statement query = this.createQueryForAppsWithOrg(orgId);
        try {
            results = this.cassandra.execute(query);
        }
        catch (Exception ex) {
            LOG.error("Failed to find Apps by Org with ID [{}]", (Object)orgId, (Object)ex);
            throw new OperationFailedException("Could not find Org's Apps: " + orgId);
        }
        List apps = Lists.create();
        for (Row row : results) {
            Application app = this.createApplicationFromRow(row);
            apps.add(app);
        }
        LOG.debug("Found {} apps in Org {}", (Object)apps.size(), (Object)orgId);
        return apps;
    }

    @Override
    public List<Application> searchByName(String searchTerm) throws TException {
        this.checkSearchTerm(searchTerm);
        throw new OperationFailedException("Searching not supported yet");
    }

    @Override
    public List<Application> getRecentlyCreated() throws TException {
        List apps = Lists.create();
        Statement query = this.createQueryForRecentlyCreatedApps();
        ResultSet results = null;
        try {
            results = this.cassandra.execute(query);
        }
        catch (Exception ex) {
            LOG.error("Failed to query for recently created apps", (Throwable)ex);
            throw new OperationFailedException("Could not get recently created apps: " + ex.getMessage());
        }
        for (Row row : results) {
            Application app = this.createApplicationFromRow(row);
            apps.add(app);
        }
        LOG.debug("Found {} recently created apps", (Object)apps.size());
        return apps;
    }

    private Statement createStatementToSave(Application app) {
        BatchStatement batch = new BatchStatement();
        UUID appId = UUID.fromString(app.applicationId);
        UUID iconId = null;
        UUID orgId = null;
        String tier = null;
        String programmingLanguage = null;
        if (app.tier != null) {
            tier = app.tier.toString();
        }
        if (app.programmingLanguage != null) {
            programmingLanguage = app.programmingLanguage.toString();
        }
        if (app.isSetApplicationIconMediaId()) {
            iconId = UUID.fromString(app.applicationIconMediaId);
        }
        if (app.isSetOrganizationId()) {
            orgId = UUID.fromString(app.organizationId);
        }
        Set owners = Sets.nullToEmpty((Set)app.owners).stream().map(UUID::fromString).collect(Collectors.toSet());
        Insert insertIntoMainTable = QueryBuilder.insertInto((String)"Applications").value("app_id", (Object)appId).value("name", (Object)app.name).value("app_description", (Object)app.applicationDescription).value("app_icon_media_id", (Object)iconId).value("organization_id", (Object)orgId).value("owners", owners).value("programming_language", (Object)programmingLanguage).value("time_provisioned", (Object)app.timeOfProvisioning).value("time_of_token_expiration", (Object)app.timeOfTokenExpiration).value("tier", (Object)tier);
        batch.add((Statement)insertIntoMainTable);
        Long timeToLive = DEFAULT_RECENT_DURATION.getSeconds();
        Insert.Options insertIntoRecentlyCreated = QueryBuilder.insertInto((String)"Applications_Recently_Created").value("app_id", (Object)appId).value("name", (Object)app.name).value("app_description", (Object)app.applicationDescription).value("app_icon_media_id", (Object)iconId).value("organization_id", (Object)orgId).value("owners", owners).value("programming_language", (Object)programmingLanguage).value("time_provisioned", (Object)app.timeOfProvisioning).value("time_of_token_expiration", (Object)app.timeOfTokenExpiration).value("tier", (Object)tier).using(QueryBuilder.ttl((int)timeToLive.intValue()));
        batch.add((Statement)insertIntoRecentlyCreated);
        return batch;
    }

    private Statement createDeleteStatementFor(Application app) {
        BatchStatement batch = new BatchStatement();
        UUID appId = UUID.fromString(app.applicationId);
        Delete.Where deleteFromMainTable = QueryBuilder.delete().all().from("Applications").where(QueryBuilder.eq((String)"app_id", (Object)appId));
        batch.add((Statement)deleteFromMainTable);
        Delete.Where deleteFromRecentsTable = QueryBuilder.delete().all().from("Applications_Recently_Created").where(QueryBuilder.eq((String)"app_id", (Object)appId));
        batch.add((Statement)deleteFromRecentsTable);
        return batch;
    }

    private Statement createQueryForAppWithId(String applicationId) {
        UUID appId = UUID.fromString(applicationId);
        return QueryBuilder.select().all().from("Applications").where(QueryBuilder.eq((String)"app_id", (Object)appId)).limit(2);
    }

    private Application createApplicationFromRow(Row row) throws OperationFailedException {
        return this.applicationMapper.apply(row);
    }

    private void checkApplicationId(String applicationId) throws InvalidArgumentException {
        Arguments.checkThat((Object)applicationId).throwing(InvalidArgumentException.class).is(RequestAssertions.validApplicationId());
    }

    private Statement createQueryToCheckIfAppIdExists(String applicationId) {
        UUID appId = UUID.fromString(applicationId);
        return QueryBuilder.select().countAll().from("Applications").where(QueryBuilder.eq((String)"app_id", (Object)appId));
    }

    private void checkRowNotMissing(String applicationId, Row row) throws ApplicationDoesNotExistException {
        Arguments.checkThat((Object)row).throwing(ApplicationDoesNotExistException.class).usingMessage("No App with ID: " + applicationId).is(Assertions.notNull());
    }

    private Statement createQueryForAppsOwnedBy(String userId) {
        UUID ownerId = UUID.fromString(userId);
        return QueryBuilder.select().all().from("Applications").where(QueryBuilder.contains((String)"owners", (Object)ownerId));
    }

    private Statement createQueryForAppsWithOrg(String orgId) {
        UUID uuid = UUID.fromString(orgId);
        return QueryBuilder.select().all().from("Applications").where(QueryBuilder.eq((String)"organization_id", (Object)uuid));
    }

    private void checkSearchTerm(String searchTerm) throws InvalidArgumentException {
        Arguments.checkThat((Object)searchTerm).throwing(InvalidArgumentException.class).is(StringAssertions.stringWithLengthGreaterThanOrEqualTo((int)2)).is(StringAssertions.nonEmptyString());
    }

    private Statement createQueryForRecentlyCreatedApps() {
        return QueryBuilder.select().all().from("Applications_Recently_Created").limit(200).allowFiltering();
    }
}

