/*
 * Decompiled with CFR 0.152.
 */
package tech.aroma.data.cassandra;

import com.datastax.driver.core.ResultSet;
import com.datastax.driver.core.Row;
import com.datastax.driver.core.Session;
import com.datastax.driver.core.Statement;
import com.datastax.driver.core.querybuilder.Insert;
import com.datastax.driver.core.querybuilder.QueryBuilder;
import java.util.Date;
import java.util.List;
import java.util.Objects;
import java.util.UUID;
import java.util.function.Function;
import java.util.stream.Collectors;
import javax.inject.Inject;
import org.apache.thrift.TBase;
import org.apache.thrift.TException;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import tech.aroma.data.ActivityRepository;
import tech.aroma.data.assertions.RequestAssertions;
import tech.aroma.thrift.LengthOfTime;
import tech.aroma.thrift.User;
import tech.aroma.thrift.events.Event;
import tech.aroma.thrift.exceptions.DoesNotExistException;
import tech.aroma.thrift.exceptions.InvalidArgumentException;
import tech.aroma.thrift.exceptions.OperationFailedException;
import tech.aroma.thrift.functions.TimeFunctions;
import tech.sirwellington.alchemy.arguments.Arguments;
import tech.sirwellington.alchemy.arguments.assertions.Assertions;
import tech.sirwellington.alchemy.arguments.assertions.BooleanAssertions;
import tech.sirwellington.alchemy.arguments.assertions.NumberAssertions;
import tech.sirwellington.alchemy.arguments.assertions.StringAssertions;
import tech.sirwellington.alchemy.thrift.ThriftObjects;

final class CassandraActivityRepository
implements ActivityRepository {
    private static final Logger LOG = LoggerFactory.getLogger(CassandraActivityRepository.class);
    private final Session session;
    private final Function<Row, Event> eventMapper;

    @Inject
    CassandraActivityRepository(Session session, Function<Row, Event> eventMapper) {
        Arguments.checkThat((Object)session, (Object[])new Object[]{eventMapper}).are(Assertions.notNull());
        this.session = session;
        this.eventMapper = eventMapper;
    }

    @Override
    public void saveEvent(Event event, User forUser, LengthOfTime lifetime) throws TException {
        this.checkEvent(event);
        this.checkUser(forUser);
        this.checkLifetime(lifetime);
        User user = forUser;
        Statement insertStatement = this.createStatementToSaveEventForUser(event, user, lifetime);
        this.tryToExecute(insertStatement, "saveEvent");
    }

    @Override
    public boolean containsEvent(String eventId, User user) throws TException {
        this.checkEventId(eventId);
        this.checkUser(user);
        Statement query = this.createQueryToCheckIfEventExists(eventId, user);
        ResultSet results = this.tryToExecute(query, "containsEvent");
        Row row = results.one();
        Arguments.checkThat((Object)row).throwing(OperationFailedException.class).usingMessage("Failed to query for event with ID " + eventId).is(Assertions.notNull());
        return row.getLong(0) > 0L;
    }

    @Override
    public Event getEvent(String eventId, User user) throws TException {
        this.checkEventId(eventId);
        this.checkUser(user);
        Statement query = this.createQueryToGetEvent(eventId, user);
        ResultSet results = this.tryToExecute(query, "getEvent");
        Row row = results.one();
        Arguments.checkThat((Object)row).throwing(DoesNotExistException.class).usingMessage("No such event with ID " + eventId + " for user " + user).is(Assertions.notNull());
        return this.mapRowToEvent(row);
    }

    @Override
    public List<Event> getAllEventsFor(User user) throws TException {
        this.checkUser(user);
        Statement query = this.createQueryToGetAllEventsForUser(user);
        ResultSet results = this.tryToExecute(query, "getAllEvents");
        return results.all().parallelStream().map(this.eventMapper::apply).filter(Objects::nonNull).collect(Collectors.toList());
    }

    @Override
    public void deleteEvent(String eventId, User user) throws TException {
        this.checkEventId(eventId);
        this.checkUser(user);
        Statement deleteStatement = this.createStatementToDelete(eventId, user);
        this.tryToExecute(deleteStatement, "deleteEvent");
    }

    @Override
    public void deleteAllEventsFor(User user) throws TException {
        this.checkUser(user);
        Statement deleteStatement = this.createStatementToDeleteAllEventsFor(user);
        this.tryToExecute(deleteStatement, "deleteAllEvents");
    }

    private void checkUser(User user) throws InvalidArgumentException {
        Arguments.checkThat((Object)user).usingMessage("user cannot be null").throwing(InvalidArgumentException.class).is(Assertions.notNull());
        Arguments.checkThat((Object)user.userId).throwing(InvalidArgumentException.class).is(RequestAssertions.validUserId());
    }

    private void checkEvent(Event event) throws InvalidArgumentException {
        Arguments.checkThat((Object)event).throwing(InvalidArgumentException.class).usingMessage("Event cannot be null").is(Assertions.notNull());
        Arguments.checkThat((Object)event.eventId).usingMessage("eventId must be a valid UUID").throwing(InvalidArgumentException.class).is(StringAssertions.validUUID());
        Arguments.checkThat((Object)event.eventType.isSet()).throwing(InvalidArgumentException.class).usingMessage("EventType must be set").is(BooleanAssertions.trueStatement());
    }

    private Statement createStatementToSaveEventForUser(Event event, User user, LengthOfTime lifetime) throws TException {
        UUID eventId = UUID.fromString(event.eventId);
        UUID userId = UUID.fromString(user.userId);
        String serializedEvent = ThriftObjects.toJson((TBase)event);
        Insert statement = QueryBuilder.insertInto((String)"Activity").value("user_id", (Object)userId).value("event_id", (Object)eventId).value("serialized_event", (Object)serializedEvent);
        if (event.isSetApplicationId()) {
            UUID appId = UUID.fromString(event.applicationId);
            statement = statement.value("app_id", (Object)appId);
        }
        if (event.isSetUserIdOfActor()) {
            UUID actorId = UUID.fromString(event.userIdOfActor);
            statement = statement.value("actor_id", (Object)actorId);
        }
        if (event.isSetTimestamp()) {
            Date timeOfEvent = new Date(event.timestamp);
            statement = statement.value("time_of_event", (Object)timeOfEvent);
        }
        int ttl = (int)TimeFunctions.toSeconds((LengthOfTime)lifetime);
        return statement.using(QueryBuilder.ttl((int)ttl));
    }

    private ResultSet tryToExecute(Statement statement, String operationName) throws OperationFailedException {
        try {
            return this.session.execute(statement);
        }
        catch (Exception ex) {
            LOG.error("Failed to execute Cassandra Statement: {}", (Object)operationName, (Object)ex);
            throw new OperationFailedException("Could not perform operation: " + ex.getMessage());
        }
    }

    private void checkEventId(String eventId) throws InvalidArgumentException {
        Arguments.checkThat((Object)eventId).throwing(InvalidArgumentException.class).usingMessage("eventId missing").is(StringAssertions.nonEmptyString()).usingMessage("eventId must be a valid uuid").is(StringAssertions.validUUID());
    }

    private Statement createQueryToCheckIfEventExists(String eventId, User user) {
        UUID eventUuid = UUID.fromString(eventId);
        UUID userUuid = UUID.fromString(user.userId);
        return QueryBuilder.select().countAll().from("Activity").where(QueryBuilder.eq((String)"user_id", (Object)userUuid)).and(QueryBuilder.eq((String)"event_id", (Object)eventUuid));
    }

    private Statement createQueryToGetEvent(String eventId, User user) {
        UUID eventUuid = UUID.fromString(eventId);
        UUID userUuid = UUID.fromString(user.userId);
        return QueryBuilder.select().all().from("Activity").where(QueryBuilder.eq((String)"user_id", (Object)userUuid)).and(QueryBuilder.eq((String)"event_id", (Object)eventUuid));
    }

    private Statement createQueryToGetAllEventsForUser(User user) {
        UUID userUuid = UUID.fromString(user.userId);
        return QueryBuilder.select().all().from("Activity").where(QueryBuilder.eq((String)"user_id", (Object)userUuid));
    }

    private Statement createStatementToDelete(String eventId, User user) {
        UUID eventUuid = UUID.fromString(eventId);
        UUID userUuid = UUID.fromString(user.userId);
        return QueryBuilder.delete().all().from("Activity").where(QueryBuilder.eq((String)"user_id", (Object)userUuid)).and(QueryBuilder.eq((String)"event_id", (Object)eventUuid));
    }

    private Statement createStatementToDeleteAllEventsFor(User user) {
        UUID userUuid = UUID.fromString(user.userId);
        return QueryBuilder.delete().all().from("Activity").where(QueryBuilder.eq((String)"user_id", (Object)userUuid));
    }

    private Event mapRowToEvent(Row row) throws DoesNotExistException {
        if (row == null) {
            return new Event();
        }
        Event event = this.eventMapper.apply(row);
        Arguments.checkThat((Object)event).usingMessage("event does not exist").throwing(DoesNotExistException.class).is(Assertions.notNull());
        return event;
    }

    private void checkLifetime(LengthOfTime lifetime) throws InvalidArgumentException {
        Arguments.checkThat((Object)lifetime).throwing(InvalidArgumentException.class).usingMessage("lifetime missing").is(Assertions.notNull());
        Arguments.checkThat((Object)lifetime.value).throwing(InvalidArgumentException.class).usingMessage("Lifetime duration must be > 0").is(NumberAssertions.greaterThan((long)0L));
    }
}

